<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;

class Skill_story_Widget extends Widget_Base {

    public function get_name() {
        return 'skill_instagram_stories';
    }

    public function get_title() {
        return __( 'استوری اینستاگرام', 'skill' );
    }

    public function get_icon() {
        return 'eicon-slider-push'; // Elementor icon
    }

    public function get_categories() {
        return [ 'skill-category' ];
    }

    public function get_keywords() {
        return [ 'instagram', 'stories', 'skill' ];
    }

    protected function register_controls() {

        // Stories Section
        $this->start_controls_section(
            'stories_section',
            [
                'label' => __( 'استوری‌ها', 'skill' ),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        // Profile Image
        $repeater->add_control(
            'profile_image',
            [
                'label' => __( 'تصویر پروفایل', 'skill' ),
                'type'  => Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );



        // Story Title
        $repeater->add_control(
            'story_title',
            [
                'label' => __( 'عنوان استوری', 'skill' ),
                'type'  => Controls_Manager::TEXT,
                'default' => __( 'استوری جدید', 'skill' ),
            ]
        );

        // Story Content Items (Nested Repeater)
        $content_repeater = new Repeater();
        
        $content_repeater->add_control(
            'content_type',
            [
                'label'   => __( 'نوع محتوا', 'skill' ),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'image' => __( 'تصویر', 'skill' ),
                    'video' => __( 'ویدیو', 'skill' ),
                    'gif'   => __( 'گیف', 'skill' ),
                ],
                'default' => 'image',
            ]
        );

        $content_repeater->add_control(
            'content_media',
            [
                'label' => __( 'فایل رسانه', 'skill' ),
                'type'  => Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $content_repeater->add_control(
            'content_duration',
            [
                'label' => __( 'مدت نمایش (ثانیه)', 'skill' ),
                'type'  => Controls_Manager::NUMBER,
                'default' => 3,
                'min'   => 1,
                'max'   => 60,
                'step'  => 0.5,
            ]
        );

        // Content Link
        $content_repeater->add_control(
            'content_link',
            [
                'label' => __( 'لینک استوری', 'skill' ),
                'type'  => Controls_Manager::URL,
                'placeholder' => __( 'https://example.com', 'skill' ),
                'default' => [
                    'url' => '',
                    'is_external' => false,
                    'nofollow' => false,
                ],
            ]
        );

        $repeater->add_control(
            'story_content_items',
            [
                'label'   => __( 'محتوای استوری', 'skill' ),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $content_repeater->get_controls(),
                'default' => [
                    [
                        'content_type' => 'image',
                        'content_media' => [
                            'url' => \Elementor\Utils::get_placeholder_image_src(),
                        ],
                        'content_duration' => 3,
                        'content_link' => [
                            'url' => '',
                            'is_external' => false,
                            'nofollow' => false,
                        ],
                    ]
                ],
                'title_field' => '{{{ content_type }}}',
            ]
        );

        $this->add_control(
            'stories_list',
            [
                'label'   => __( 'لیست استوری‌ها', 'skill' ),
                'type'    => Controls_Manager::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => [
                    [
                        'profile_image' => [
                            'url' => \Elementor\Utils::get_placeholder_image_src(),
                        ],
                        'story_title' => __( 'استوری جدید', 'skill' ),
                        'story_content_items' => [
                            [
                                'content_type' => 'image',
                                'content_media' => [
                                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                                ],
                                'content_duration' => 3,
                                'content_link' => [
                                    'url' => '',
                                    'is_external' => false,
                                    'nofollow' => false,
                                ],
                            ]
                        ],
                    ]
                ],
                'title_field' => '{{{ story_title }}}',
            ]
        );

        // Story Title Color Control
        $this->add_control(
            'story_title_color',
            [
                'label' => __( 'رنگ عنوان استوری', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .story-title' => 'color: {{VALUE}};',
                ],
                'default' => '#333333',
            ]
        );

        $this->end_controls_section();

        // Display toggles
        $this->start_controls_section(
            'display_section',
            [
                'label' => __( 'نمایش عناصر', 'skill' ),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'show_share',
            [
                'label' => __( 'نمایش اشتراک‌گذاری', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __( 'بله', 'skill' ),
                'label_off' => __( 'خیر', 'skill' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_like',
            [
                'label' => __( 'نمایش لایک و تعداد لایک', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __( 'بله', 'skill' ),
                'label_off' => __( 'خیر', 'skill' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        if ( ! empty( $settings['stories_list'] ) ) :
            ?>
            <div class="skill-instagram-stories">
                <?php 
                $show_share = isset($settings['show_share']) ? ($settings['show_share'] === 'yes') : true;
                $show_like  = isset($settings['show_like']) ? ($settings['show_like'] === 'yes') : true;
                ?>
                <div class="stories-wrapper">
                    <?php foreach ( $settings['stories_list'] as $story ) : ?>
                        <div class="story-item" data-story='<?php echo esc_attr(json_encode($story)); ?>'>
                            <div class="story-thumbnail">
                                <?php 
                                // Use profile image as thumbnail
                                $profile_url = !empty($story['profile_image']['url']) ? $story['profile_image']['url'] : \Elementor\Utils::get_placeholder_image_src();
                                ?>
                                <img src="<?php echo esc_url($profile_url); ?>" alt="<?php echo esc_attr( $story['story_title'] ); ?>">
                            </div>
                            <span class="story-title"><?php echo esc_html( $story['story_title'] ); ?></span>
                        </div>
                    <?php endforeach; ?>
                </div>
                <div class="story-popup">
                    <span class="close-popup">&times;</span>
                    <div class="story-header">
                        <div class="story-profile-info">
                            <div class="story-profile-image">
                                <img src="" alt="Profile" class="header-profile-img">
                            </div>
                            <span class="story-profile-name"></span>
                        </div>
                        <div class="story-progress-bars">
                            <div class="progress-bar"><div class="progress-fill" data-index="0"></div></div>
                        </div>
                        <div class="story-time">اکنون</div>
                    </div>
                    <div class="story-content">
                    </div>
                    <div class="story-title-overlay" aria-hidden="true"></div>
                    <?php if ( $show_share || $show_like ) : ?>
                    <div class="story-share">
                        <div class="like-and-share">
                            <?php if ( $show_like ) : ?>
                                <button type="button" class="like-toggle" aria-pressed="false" title="پسندیدن"></button>
                                <span class="like-count" aria-live="polite" title="تعداد لایک">0</span>
                            <?php endif; ?>
                            <?php if ( $show_share ) : ?>
                                <button type="button" class="share-toggle">اشتراک‌گذاری</button>
                            <?php endif; ?>
                        </div>
                        <?php if ( $show_share ) : ?>
                        <div class="share-menu" aria-hidden="true">
                            <a href="#" class="share-icon share-telegram" title="تلگرام" target="_blank" rel="nofollow noopener"></a>
                            <a href="#" class="share-icon share-whatsapp" title="واتساپ" target="_blank" rel="nofollow noopener"></a>
                        </div>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>
                    <div class="story-right-area"></div>
                    <div class="story-left-area"></div>
                </div>
            </div>

            <style>
                /* --- Instagram Stories Styles --- */
                .skill-instagram-stories {
                    position: relative;
                    width: 100%;
                }
                .stories-wrapper {
                    display: flex;
                    gap: 22px;
                    overflow-x: auto;
                    padding: 10px;
                    scrollbar-width: none;
                }
                .stories-wrapper::-webkit-scrollbar {
                    display: none;
                }
                .story-item {
                    text-align: center;
                    cursor: pointer;
                    flex-shrink: 0;
                }
                .story-thumbnail {
                    width: 88px;
                    height: 88px;
                    border-radius: 50%;
                    overflow: hidden;
                    border: 3px solid transparent;
                    background: linear-gradient(45deg, #f09433 0%, #e6683c 25%, #dc2743 50%, #cc2366 75%, #bc1888 100%);
                    background-clip: padding-box;
                    padding: 3px;
                    margin: 0 auto 5px;
                }
                .story-thumbnail img {
                    width: 100%;
                    height: 100%;
                    object-fit: cover;
                    border-radius: 50%;
                }
                .story-title {
                    font-size: 16px;
                    display: block;
                    color: #333333; /* Default color, will be overridden by Elementor control */
                    transition: color 0.3s ease;
                    max-width: 100px;
                    overflow: hidden;
                    text-overflow: ellipsis;
                    white-space: nowrap;
                }
                .story-popup {
                    display: none;
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(0,0,0,0.8);
                    justify-content: center;
                    align-items: center;
                    z-index: 9999;
                    flex-direction: column;
                }
                .story-popup.active {
                    display: flex;
                }
                .story-content {
                    max-width: 100%;
                    max-height: 100%;
                    position: relative;
                    border-radius: 10px;
                    overflow: hidden;
                    cursor: pointer;
                }
                .story-content img, .story-content video {
                    max-width: 100%;
                    max-height: 100%;
                    border-radius: 10px;
                    width: 375px;
                    height: 667px;
                    object-fit: cover;
                    display: block;
                }
                .story-content a {
                    display: block;
                    text-decoration: none;
                    color: inherit;
                }
                .story-content a:hover {
                    opacity: 0.9;
                }
                .story-title-overlay {
                    position: absolute;
                    top: 50%;
                    left: 50%;
                    transform: translate(-50%, -50%);
                    color: #fff;
                    font-size: 22px;
                    font-weight: 700;
                    text-align: center;
                    padding: 10px 16px;
                    border-radius: 10px;
                    background: rgba(0,0,0,0.35);
                    backdrop-filter: blur(3px);
                    opacity: 0;
                    pointer-events: none;
                    transition: opacity 400ms ease;
                    z-index: 10002;
                    max-width: 80vw;
                    white-space: nowrap;
                    overflow: hidden;
                    text-overflow: ellipsis;
                }
                .story-title-overlay.active { opacity: 1; }
                .story-share {
                    position: absolute;
                    bottom: 20px;
                    left: 50%;
                    transform: translateX(-50%);
                    display: flex;
                    flex-direction: column-reverse;
                    align-items: center;
                    gap: 10px;
                    z-index: 10001;
                }
                .like-and-share { display: flex; gap: 8px; align-items: center; }
                .like-toggle {
                    width: 40px;
                    height: 40px;
                    border-radius: 50%;
                    border: none;
                    cursor: pointer;
                    background: rgba(0,0,0,0.5);
                    backdrop-filter: blur(4px);
                    position: relative;
                }
                .like-toggle::before{
                    content: '';
                    position: absolute;
                    top: 50%;
                    left: 50%;
                    width: 22px;
                    height: 22px;
                    transform: translate(-50%, -50%);
                    background-repeat: no-repeat;
                    background-position: center;
                    background-size: contain;
                    background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="white" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"/></svg>');
                }
                .like-toggle.liked { background: rgba(255,255,255,0.15); }
                .like-toggle.liked::before{
                    background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path fill="%23ff2d55" d="M12 21.35l-1.45-1.32C5.4 15.36 2 12.28 2 8.5 2 6 4 4 6.5 4c1.74 0 3.41.81 4.5 2.09C12.09 4.81 13.76 4 15.5 4 18 4 20 6 20 8.5c0 3.78-3.4 6.86-8.55 11.54L12 21.35z"/></svg>');
                }
                .like-count {
                    color: #fff;
                    font-size: 13px;
                    min-width: 22px;
                    text-align: center;
                    background: rgba(255,255,255,0.15);
                    padding: 6px 10px;
                    border-radius: 999px;
                    line-height: 1;
                    user-select: none;
                }
                .share-toggle {
                    background: rgba(0,0,0,0.5);
                    color: #fff;
                    border: none;
                    border-radius: 999px;
                    padding: 8px 14px;
                    font-size: 13px;
                    cursor: pointer;
                    backdrop-filter: blur(4px);
                }
                .share-menu {
                    display: none;
                    background: rgba(0,0,0,0.6);
                    padding: 8px 10px;
                    border-radius: 12px;
                    display: none;
                    gap: 10px;
                }
                .story-share.open .share-menu { display: flex; }
                .share-icon {
                    width: 40px;
                    height: 40px;
                    border-radius: 50%;
                    display: inline-block;
                    background-position: center;
                    background-repeat: no-repeat;
                    background-size: 22px 22px;
                }
                .share-telegram { background-color: #229ED9; background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 240 240" fill="white"><path d="M120 0C53.8 0 0 53.8 0 120s53.8 120 120 120 120-53.8 120-120S186.2 0 120 0zm58.7 80.6l-19.9 94.1c-1.5 6.5-5.3 8.1-10.7 5l-29.6-21.8-14.3 13.8c-1.6 1.6-3 3-6.1 3l2.2-31.6 57.5-51.9c2.5-2.2-.5-3.5-3.9-1.2l-71 44.7-30.6-9.6c-6.6-2.1-6.7-6.6 1.4-9.8l119.5-46.1c5.5-2 10.3 1.3 8.6 9.4z"/></svg>'); }
                .share-whatsapp { background-color: #25D366; background-image: url('data:image/svg+xml;utf8,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 32 32" fill="white"><path d="M19.11 17.41c-.27-.14-1.6-.79-1.85-.88-.25-.09-.43-.14-.62.14-.19.27-.71.88-.87 1.06-.16.18-.32.2-.59.07-.27-.14-1.14-.42-2.18-1.33-.81-.72-1.36-1.6-1.52-1.87-.16-.27-.02-.42.12-.56.12-.12.27-.32.41-.48.14-.16.18-.27.27-.45.09-.18.05-.34-.02-.48-.07-.14-.62-1.5-.85-2.05-.22-.53-.45-.45-.62-.46l-.53-.01c-.18 0-.48.07-.73.34-.25.27-.96.94-.96 2.29s.98 2.66 1.12 2.84c.14.18 1.93 2.95 4.68 4.14.65.28 1.15.45 1.54.58.65.2 1.24.17 1.71.1.52-.08 1.6-.65 1.83-1.28.23-.63.23-1.18.16-1.28-.07-.1-.25-.16-.52-.29z"/><path d="M26.64 5.36A12.7 12.7 0 0 0 16 1.33C8.94 1.33 3.2 7.07 3.2 14.13c0 2.3.62 4.45 1.7 6.3L3 30.67l10.47-1.9c1.77.97 3.8 1.52 5.97 1.52 7.06 0 12.8-5.74 12.8-12.8 0-3.41-1.36-6.58-3.6-8.93zM16 28c-2.03 0-3.92-.62-5.49-1.68l-.39-.25-6.13 1.11 1.16-5.99-.26-.41a11.33 11.33 0 1 1 11.11 7.22z"/></svg>'); }
                .story-header {
                    position: absolute;
                    top: 20px;
                    left: 20px;
                    right: 20px;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                    z-index: 10000;
                }
                .story-profile-info {
                    display: flex;
                    align-items: center;
                    gap: 12px;
                }
                .story-profile-image {
                    width: 40px;
                    height: 40px;
                    border-radius: 50%;
                    overflow: hidden;
                    border: 2px solid #fff;
                }
                .story-profile-image img {
                    width: 100%;
                    height: 100%;
                    object-fit: cover;
                }
                .story-profile-name {
                    color: #fff;
                    font-weight: 600;
                    font-size: 16px;
                }
                .story-progress-bars {
                    display: flex;
                    gap: 3px;
                    flex: 1;
                    margin: 0 20px;
                }
                .story-time {
                    color: rgba(255,255,255,0.8);
                    font-size: 14px;
                    white-space: nowrap;
                }
                .progress-bar {
                    flex: 1;
                    height: 2px;
                    background: rgba(255,255,255,0.3);
                    border-radius: 2px;
                    overflow: hidden;
                    margin: 0 1px;
                    min-width: 20px;
                }
                .progress-fill {
                    height: 100%;
                    background: #fff;
                    width: 0%;
                    transition: width linear;
                    border-radius: 2px;
                }
                .close-popup {
                    position: absolute;
                    top: -5px;
                    right: 5px;
                    font-size: 40px;
                    color: #fff;
                    cursor: pointer;
                    z-index: 10001;
                }
                .story-left-area, .story-right-area {
                    position: absolute;
                    top: 0;
                    width: 50%;
                    height: 100%;
                    cursor: pointer;
                    z-index: 10000;
                }
                .story-left-area {
                    right: -90px;
                }
                .story-right-area {
                    left: -90px;
                }

                /* --- Responsive Styles --- */
                
                /* Tablet Styles */
                @media (max-width: 768px) {
                    .stories-wrapper {
                        gap: 12px;
                        padding: 8px;
                    }
                    .story-thumbnail {
                        width: 72px;
                        height: 72px;
                    }
                    .story-title {
                        font-size: 13px;
                        max-width: 85px;
                    }
                    .story-content img, .story-content video {
                        width: 320px;
                        height: 568px;
                    }
                    .story-header {
                        top: 15px;
                        left: 15px;
                        right: 15px;
                        padding: 12px 15px;
                    }
                    .story-profile-image {
                        width: 35px;
                        height: 35px;
                    }
                    .story-profile-name {
                        font-size: 14px;
                    }
                    .story-time {
                        font-size: 12px;
                    }
                    .close-popup {
                        top: 0;
                        right: -7px;
                        font-size: 35px;
                    }
                }

                /* Mobile Styles */
                @media (max-width: 480px) {
                    .stories-wrapper {
                        gap: 10px;
                        padding: 6px;
                    }
                    .story-thumbnail {
                        width: 60px;
                        height: 60px;
                    }
                    .story-title {
                        font-size: 12px;
                        max-width: 75px;
                    }
                    .story-content img, .story-content video {
                        width: 280px;
                        height: 497px;
                    }
                    .story-share { bottom: 12px; gap: 8px; }
                    .story-header {
                        top: 10px;
                        left: 10px;
                        right: 10px;
                        padding: 10px 12px;
                        flex-direction: column;
                        gap: 8px;
                        align-items: center;
                    }
                    .story-left-area {
                        right: -60px;
                    }
                    .story-right-area {
                        left: -60px;
                    }
                    .story-profile-info {
                        gap: 8px;
                    }
                    .story-profile-image {
                        width: 30px;
                        height: 30px;
                    }
                    .story-profile-name {
                        font-size: 13px;
                    }
                    .story-progress-bars {
                        margin: 0;
                        width: 100%;
                        order: 2;
                    }
                    .story-time {
                        font-size: 11px;
                        order: 3;
                    }
                    .close-popup {
                        top: 5px;
                        right: 15px;
                        font-size: 30px;
                    }
                    .story-left-area, .story-right-area {
                        width: 40%;
                    }
                }

                /* Small Mobile Styles */
                @media (max-width: 360px) {
                    .stories-wrapper {
                        gap: 8px;
                        padding: 5px;
                    }
                    .story-thumbnail {
                        width: 52px;
                        height: 52px;
                    }
                    .story-title {
                        font-size: 11px;
                        max-width: 65px;
                    }
                    .story-content img, .story-content video {
                        width: 250px;
                        height: 444px;
                    }
                    .story-header {
                        top: 8px;
                        left: 8px;
                        right: 8px;
                        padding: 8px 10px;
                    }
                    .story-profile-image {
                        width: 28px;
                        height: 28px;
                    }
                    .story-profile-name {
                        font-size: 12px;
                    }
                    .story-time {
                        font-size: 10px;
                    }
                    .close-popup {
                        top: 3px;
                        right: 12px;
                        font-size: 28px;
                    }
                }

                /* Landscape Mobile Styles */
                @media (max-width: 768px) and (orientation: landscape) {
                    .story-header {
                        top: 10px;
                        left: 10px;
                        right: 10px;
                        padding: 8px 15px;
                    }
                }

                /* High DPI Displays */
                @media (-webkit-min-device-pixel-ratio: 2), (min-resolution: 192dpi) {
                    .story-thumbnail {
                        border-width: 2px;
                    }
                    .story-profile-image {
                        border-width: 1px;
                    }
                }

                /* Body scroll lock for mobile */
                body.story-open {
                    overflow: hidden;
                    position: fixed;
                    width: 100%;
                }

                /* Touch-friendly improvements */
                @media (hover: none) and (pointer: coarse) {
                    .story-item {
                        min-height: 44px;
                        min-width: 44px;
                    }
                    .close-popup {
                        min-height: 44px;
                        min-width: 44px;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                    }
                    .story-left-area, .story-right-area {
                        min-height: 44px;
                    }
                }
            </style>

            <script>
                (function($){
                    $(document).ready(function(){
                        var currentStory = null;
                        var currentItemIndex = 0;
                        var storyTimer = null;
                        var titleOverlayTimer = null;
                        var touchStartX = 0;
                        var touchStartY = 0;
                        var isMobile = window.innerWidth <= 768;
                        var currentStoryKey = null;
                        var allStories = [];
                        var currentStoryIndex = 0;

                        function updateShareLinks(item) {
                            if (!currentStory) return;
                            var title = currentStory.story_title || '';
                            var targetUrl = '';
                            if (item && item.content_link && item.content_link.url) {
                                targetUrl = item.content_link.url;
                            } else if (item && item.content_media && item.content_media.url) {
                                targetUrl = item.content_media.url;
                            } else if (currentStory.profile_image && currentStory.profile_image.url) {
                                targetUrl = currentStory.profile_image.url;
                            } else {
                                targetUrl = window.location.href;
                            }

                            var encUrl = encodeURIComponent(targetUrl);
                            var encText = encodeURIComponent(title);

                            var telegramHref = 'https://t.me/share/url?url=' + encUrl + '&text=' + encText;
                            var whatsappHref = 'https://api.whatsapp.com/send?text=' + encText + '%20' + encUrl;

                            $('.story-share .share-telegram').attr('href', telegramHref);
                            $('.story-share .share-whatsapp').attr('href', whatsappHref);
                        }

                        function buildStoryKey(storyData){
                            var title = (storyData && storyData.story_title) ? String(storyData.story_title) : '';
                            var img = (storyData && storyData.profile_image && storyData.profile_image.url) ? String(storyData.profile_image.url) : '';
                            return btoa(unescape(encodeURIComponent(title + '|' + img)));
                        }

                        function getItemLikeCount(itemIndex){
                            if (!currentStoryKey) return 0;
                            var likeKey = 'skill_story_likes_' + btoa(currentStoryKey + '_' + itemIndex);
                            var raw = localStorage.getItem(likeKey);
                            var n = parseInt(raw, 10);
                            return isNaN(n) ? 0 : n;
                        }

                        function setItemLikeCount(itemIndex, value){
                            if (!currentStoryKey) return;
                            var likeKey = 'skill_story_likes_' + btoa(currentStoryKey + '_' + itemIndex);
                            try { localStorage.setItem(likeKey, String(Math.max(0, value|0))); } catch(err) {}
                        }

                        function isItemLiked(itemIndex){
                            if (!currentStoryKey) return false;
                            var userKey = 'skill_story_users_' + btoa(currentStoryKey + '_' + itemIndex);
                            return localStorage.getItem(userKey) === '1';
                        }

                        function setItemLiked(itemIndex, liked){
                            if (!currentStoryKey) return;
                            var userKey = 'skill_story_users_' + btoa(currentStoryKey + '_' + itemIndex);
                            try { 
                                if (liked) {
                                    localStorage.setItem(userKey, '1');
                                } else {
                                    localStorage.removeItem(userKey);
                                }
                            } catch(err) {}
                        }

                        function updateLikeCountDisplay(itemIndex){
                            var count = getItemLikeCount(itemIndex);
                            $('.like-count').text(String(count));
                        }

                        function refreshLikeState(itemIndex){
                            var liked = isItemLiked(itemIndex);
                            var $btn = $('.like-toggle');
                            $btn.toggleClass('liked', !!liked);
                            $btn.attr('aria-pressed', liked ? 'true' : 'false');
                            updateLikeCountDisplay(itemIndex);
                        }


                        $('.story-item').on('click', function(){
                            // Collect all stories from the page
                            allStories = [];
                            $('.story-item').each(function(index) {
                                allStories.push($(this).data('story'));
                            });
                            
                            // Find current story index
                            var storyData = $(this).data('story');
                            currentStoryIndex = $(this).index();
                            
                            openStory(storyData);
                        });

                        // Toggle share menu
                        $(document).on('click', '.share-toggle', function(e) {
                            e.stopPropagation();
                            var $wrap = $(this).closest('.story-share');
                            $wrap.toggleClass('open');
                            $wrap.find('.share-menu').attr('aria-hidden', !$wrap.hasClass('open'));
                        });

                        // Like click (toggle like/unlike per item using AJAX)
                        $(document).on('click', '.like-toggle', function(e){
                            e.stopPropagation();
                            if (!currentStory) return;
                            if (!currentStoryKey) currentStoryKey = buildStoryKey(currentStory);
                            
                            var $btn = $(this);
                            var likedAlready = isItemLiked(currentItemIndex);
                            var action = likedAlready ? 'unlike' : 'like';
                            
                            // Check if AJAX data is available
                            if (typeof skill_story_like_data === 'undefined') {
                                console.log('AJAX data not available, using local storage only');
                                // Fallback to local storage only
                                if (likedAlready) {
                                    var curr = getItemLikeCount(currentItemIndex);
                                    setItemLikeCount(currentItemIndex, Math.max(0, curr - 1));
                                    setItemLiked(currentItemIndex, false);
                                } else {
                                    var curr2 = getItemLikeCount(currentItemIndex);
                                    setItemLikeCount(currentItemIndex, curr2 + 1);
                                    setItemLiked(currentItemIndex, true);
                                }
                                refreshLikeState(currentItemIndex);
                                return;
                            }
                            
                            // Disable button during request
                            $btn.prop('disabled', true);
                            
                            // Make AJAX request
                            $.ajax({
                                url: skill_story_like_data.ajaxurl,
                                type: 'POST',
                                data: {
                                    action: 'skill_story_like',
                                    nonce: skill_story_like_data.nonce,
                                    story_key: currentStoryKey,
                                    item_index: currentItemIndex,
                                    action_type: action
                                },
                                success: function(response) {
                                    console.log('AJAX response:', response);
                                    if (response.success) {
                                        // Update local storage with server data
                                        setItemLikeCount(currentItemIndex, response.data.likes_count);
                                        setItemLiked(currentItemIndex, response.data.user_has_liked);
                                        
                                        // Update UI
                                        refreshLikeState(currentItemIndex);
                                        
                                        // Show success message briefly
                                        var $message = $('.like-count');
                                        var originalText = $message.text();
                                        $message.text(response.data.message).css('color', '#4CAF50');
                                        setTimeout(function() {
                                            $message.text(response.data.likes_count).css('color', '');
                                        }, 2000);
                                    } else {
                                        console.log('Server error:', response.data);
                                        // Show error message
                                        var $message = $('.like-count');
                                        var originalText = $message.text();
                                        $message.text(response.data.message || 'خطا در ثبت لایک').css('color', '#f44336');
                                        setTimeout(function() {
                                            $message.text(originalText).css('color', '');
                                        }, 3000);
                                    }
                                },
                                error: function(xhr, status, error) {
                                    console.log('AJAX error:', error, xhr.responseText);
                                    // Show error message
                                    var $message = $('.like-count');
                                    var originalText = $message.text();
                                    $message.text('خطا در اتصال به سرور').css('color', '#f44336');
                                    setTimeout(function() {
                                        $message.text(originalText).css('color', '');
                                    }, 3000);
                                },
                                complete: function() {
                                    // Re-enable button
                                    $btn.prop('disabled', false);
                                }
                            });
                        });

                        function openStory(storyData) {
                            currentStory = storyData;
                            currentItemIndex = 0;
                            currentStoryKey = buildStoryKey(storyData);
                            
                            // Debug: Check if AJAX data is available
                            console.log('AJAX data available:', typeof skill_story_like_data !== 'undefined');
                            if (typeof skill_story_like_data !== 'undefined') {
                                console.log('AJAX URL:', skill_story_like_data.ajaxurl);
                                console.log('Nonce:', skill_story_like_data.nonce);
                            }
                            
                            // Update story header
                            $('.header-profile-img').attr('src', storyData.profile_image.url);
                            $('.story-profile-name').text(storyData.story_title);
                            
                            // Load like data from server for all items
                            loadLikeDataFromServer();
                            
                            // Show centered title overlay briefly
                            var $overlay = $('.story-title-overlay');
                            $overlay.text(storyData.story_title || '');
                            $overlay.addClass('active').attr('aria-hidden', 'false');
                            if (titleOverlayTimer) { clearTimeout(titleOverlayTimer); }
                            titleOverlayTimer = setTimeout(function(){
                                $overlay.removeClass('active').attr('aria-hidden', 'true');
                            }, 2000);
                            
                            if (!storyData.story_content_items || storyData.story_content_items.length === 0) {
                                // Fallback to profile image if no content items
                                var content = '<img src="'+storyData.profile_image.url+'" alt="Story Profile">';
                                $('.story-content').html(content);
                                // Create single progress bar for single media
                                $('.story-header .story-progress-bars').html('<div class="progress-bar"><div class="progress-fill" data-index="0"></div></div>');
                                // Reset progress bar to 0% before starting animation
                                $('.progress-fill[data-index="0"]').css('width', '0%');
                                startProgressAnimation(0);
                                // Update share links for fallback
                                var fallbackItem = { content_link: { url: '' }, content_media: { url: storyData.profile_image.url } };
                                updateShareLinks(fallbackItem);
                                refreshLikeState(0);
                            } else {
                                // New multi-content format
                                createProgressBars();
                                showStoryItem(0);
                            }
                            
                            $('.story-popup').addClass('active');
                            
                            // Prevent body scroll on mobile
                            if (isMobile) {
                                $('body').addClass('story-open');
                            }
                        }

                        function loadLikeDataFromServer() {
                            if (!currentStory || !currentStoryKey) return;
                            
                            // Check if AJAX data is available
                            if (typeof skill_story_like_data === 'undefined') {
                                console.log('AJAX data not available, using local storage only');
                                refreshLikeState(currentItemIndex);
                                return;
                            }
                            
                            var items = currentStory.story_content_items || [{ content_media: { url: currentStory.profile_image.url } }];
                            
                            // Load like data for each item
                            items.forEach(function(item, index) {
                                $.ajax({
                                    url: skill_story_like_data.ajaxurl,
                                    type: 'POST',
                                    data: {
                                        action: 'skill_story_like',
                                        nonce: skill_story_like_data.nonce,
                                        story_key: currentStoryKey,
                                        item_index: index,
                                        action_type: 'get_data'
                                    },
                                    success: function(response) {
                                        if (response.success) {
                                            // Update local storage with server data
                                            setItemLikeCount(index, response.data.likes_count);
                                            setItemLiked(index, response.data.user_has_liked);
                                            
                                            // Update UI if this is the current item
                                            if (index === currentItemIndex) {
                                                refreshLikeState(index);
                                            }
                                        } else {
                                            console.log('Server response error:', response.data);
                                            refreshLikeState(index);
                                        }
                                    },
                                    error: function(xhr, status, error) {
                                        console.log('AJAX error:', error);
                                        // Fallback to local storage if server fails
                                        refreshLikeState(index);
                                    }
                                });
                            });
                        }

                        function createProgressBars() {
                            var progressHtml = '';
                            for(var i = 0; i < currentStory.story_content_items.length; i++) {
                                progressHtml += '<div class="progress-bar"><div class="progress-fill" data-index="'+i+'"></div></div>';
                            }
                            $('.story-header .story-progress-bars').html(progressHtml);
                            
                            // Wait for DOM to render, then reset progress bars
                            setTimeout(function() {
                                // Reset all progress bars to 0% with no transition
                                $('.progress-fill').css('transition', 'none').css('width', '0%');
                            }, 10);
                        }

                        function showStoryItem(index) {
                            if (!currentStory.story_content_items || index >= currentStory.story_content_items.length) {
                                // Current story finished, try to go to next story
                                goToNextStory();
                                return;
                            }

                            currentItemIndex = index;
                            var item = currentStory.story_content_items[index];
                            var content = '';

                            if(item.content_type === 'video') {
                                content = '<video controls autoplay muted><source src="'+item.content_media.url+'" type="video/mp4"></video>';
                            } else {
                                content = '<img src="'+item.content_media.url+'" alt="Story Content">';
                            }

                            // Add link wrapper if content_link exists
                            if(item.content_link && item.content_link.url) {
                                var linkTarget = item.content_link.is_external ? 'target="_blank"' : '';
                                var linkRel = item.content_link.nofollow ? 'rel="nofollow"' : '';
                                content = '<a href="'+item.content_link.url+'" '+linkTarget+' '+linkRel+'>'+content+'</a>';
                            }

                            // Update only the content area, preserve header
                            $('.story-content').html(content);
                            // Update share links and like state for current item
                            updateShareLinks(item);
                            refreshLikeState(currentItemIndex);
                            
                            // Wait a bit for DOM to render, then reset progress bars
                            setTimeout(function() {
                                // Reset all progress bars to 0%
                                $('.progress-fill').css('transition', 'none').css('width', '0%');
                                
                                // Fill completed progress bars
                                for(var i = 0; i < index; i++) {
                                    $('.progress-fill[data-index="'+i+'"]').css('transition', 'none').css('width', '100%');
                                }
                                
                                // Start progress animation for current item
                                startProgressAnimation(index);
                            }, 10);
                            
                            // Auto advance to next item
                            var duration = (item.content_duration || 3) * 1000;
                            storyTimer = setTimeout(function() {
                                showStoryItem(index + 1);
                            }, duration);
                        }

                        function startProgressAnimation(index) {
                            var item = currentStory.story_content_items[index];
                            var duration = (item.content_duration || 3) * 1000;
                            
                            // Get the progress bar element
                            var progressBar = $('.progress-fill[data-index="'+index+'"]');
                            
                            // Reset transition and width first
                            progressBar.css('transition', 'none').css('width', '0%');
                            
                            // Force reflow to ensure CSS changes are applied
                            progressBar[0].offsetHeight;
                            
                            // Now set transition and animate
                            progressBar.css('transition', 'width '+duration+'ms linear').css('width', '100%');
                        }

                        function goToNextStory() {
                            // Check if there's a next story
                            if (currentStoryIndex < allStories.length - 1) {
                                currentStoryIndex++;
                                
                                // Clear current timers
                                if (storyTimer) {
                                    clearTimeout(storyTimer);
                                    storyTimer = null;
                                }
                                
                                // Open next story
                                var nextStory = allStories[currentStoryIndex];
                                openStory(nextStory);
                            } else {
                                closeStory();
                            }
                        }

                        function closeStory() {
                            $('.story-popup').removeClass('active');
                            $('.story-content').html('');
                            // Reset header
                            $('.header-profile-img').attr('src', '');
                            $('.story-profile-name').text('');
                            // Hide and clear title overlay
                            var $overlay = $('.story-title-overlay');
                            $overlay.removeClass('active').attr('aria-hidden', 'true').text('');
                            if (titleOverlayTimer) { clearTimeout(titleOverlayTimer); titleOverlayTimer = null; }
                            if (storyTimer) {
                                clearTimeout(storyTimer);
                                storyTimer = null;
                            }
                            currentStory = null;
                            currentItemIndex = 0;
                            currentStoryIndex = 0;
                            allStories = [];
                            
                            // Restore body scroll on mobile
                            if (isMobile) {
                                $('body').removeClass('story-open');
                            }
                        }

                        $('.close-popup').on('click', closeStory);

                        // Touch events for mobile
                        $('.story-content').on('touchstart', function(e) {
                            touchStartX = e.originalEvent.touches[0].clientX;
                            touchStartY = e.originalEvent.touches[0].clientY;
                        });

                        $('.story-content').on('touchend', function(e) {
                            if (!currentStory) return;
                            
                            var touchEndX = e.originalEvent.changedTouches[0].clientX;
                            var touchEndY = e.originalEvent.changedTouches[0].clientY;
                            var diffX = touchStartX - touchEndX;
                            var diffY = touchStartY - touchEndY;
                            
                            // Minimum swipe distance
                            var minSwipeDistance = 50;
                            
                            if (Math.abs(diffX) > Math.abs(diffY) && Math.abs(diffX) > minSwipeDistance) {
                                if (diffX > 0) {
                                    // Swipe left - next item or next story
                                    if (storyTimer) {
                                        clearTimeout(storyTimer);
                                    }
                                    if (currentStory && currentStory.story_content_items && currentItemIndex < currentStory.story_content_items.length - 1) {
                                        // Go to next item in current story
                                        showStoryItem(currentItemIndex + 1);
                                    } else {
                                        // Go to next story
                                        goToNextStory();
                                    }
                                } else {
                                    // Swipe right - previous item or previous story
                                    if (storyTimer) {
                                        clearTimeout(storyTimer);
                                    }
                                    if (currentItemIndex > 0) {
                                        // Go to previous item in current story
                                        showStoryItem(currentItemIndex - 1);
                                    } else if (currentStoryIndex > 0) {
                                        // Go to previous story
                                        currentStoryIndex--;
                                        var prevStory = allStories[currentStoryIndex];
                                        openStory(prevStory);
                                    }
                                }
                            }
                        });

                        // Click on left side to go to previous story item or previous story
                        $('.story-left-area').on('click', function(){
                            if (storyTimer) {
                                clearTimeout(storyTimer);
                            }
                            if (currentItemIndex > 0) {
                                // Go to previous item in current story
                                showStoryItem(currentItemIndex - 1);
                            } else if (currentStoryIndex > 0) {
                                // Go to previous story
                                currentStoryIndex--;
                                var prevStory = allStories[currentStoryIndex];
                                openStory(prevStory);
                            }
                        });

                        // Click on right side to go to next story item or next story
                        $('.story-right-area').on('click', function(){
                            if (storyTimer) {
                                clearTimeout(storyTimer);
                            }
                            if (currentStory && currentStory.story_content_items && currentItemIndex < currentStory.story_content_items.length - 1) {
                                // Go to next item in current story
                                showStoryItem(currentItemIndex + 1);
                            } else {
                                // Go to next story
                                goToNextStory();
                            }
                        });

                        // Click on story content to go to next item or next story
                        $('.story-content').on('click', function(e){
                            // Don't trigger on touch devices to avoid conflict with swipe
                            if (e.type === 'click' && 'ontouchstart' in window) return;
                            
                            if (storyTimer) {
                                clearTimeout(storyTimer);
                            }
                            if (currentStory && currentStory.story_content_items && currentItemIndex < currentStory.story_content_items.length - 1) {
                                // Go to next item in current story
                                showStoryItem(currentItemIndex + 1);
                            } else {
                                // Go to next story
                                goToNextStory();
                            }
                        });

                        // Keyboard navigation
                        $(document).on('keydown', function(e){
                            if (!$('.story-popup').hasClass('active')) return;
                            
                            if (e.keyCode === 37) { // Left arrow
                                if (storyTimer) {
                                    clearTimeout(storyTimer);
                                }
                                if (currentItemIndex > 0) {
                                    // Go to previous item in current story
                                    showStoryItem(currentItemIndex - 1);
                                } else if (currentStoryIndex > 0) {
                                    // Go to previous story
                                    currentStoryIndex--;
                                    var prevStory = allStories[currentStoryIndex];
                                    openStory(prevStory);
                                }
                            } else if (e.keyCode === 39) { // Right arrow
                                if (storyTimer) {
                                    clearTimeout(storyTimer);
                                }
                                if (currentStory && currentStory.story_content_items && currentItemIndex < currentStory.story_content_items.length - 1) {
                                    // Go to next item in current story
                                    showStoryItem(currentItemIndex + 1);
                                } else {
                                    // Go to next story
                                    goToNextStory();
                                }
                            } else if (e.keyCode === 27) { // Escape
                                closeStory();
                            }
                        });

                        // Handle window resize
                        $(window).on('resize', function() {
                            isMobile = window.innerWidth <= 768;
                        });

                        // Close share menu and story on outside click / escape
                        $(document).on('click', function(e) {
                            if ($('.story-popup').hasClass('active') && !$(e.target).closest('.story-popup, .story-item').length) {
                                closeStory();
                            }
                            if (!$(e.target).closest('.story-share').length) {
                                $('.story-share').removeClass('open').find('.share-menu').attr('aria-hidden', 'true');
                            }
                        });
                    });
                    // scorol story
                    $('.stories-wrapper').on('mousedown touchstart', function(e) {
                        var startX = e.pageX || e.originalEvent.touches[0].pageX;
                        var scrollLeft = $(this).scrollLeft();
                        var $this = $(this);
    
                        $(document).on('mousemove touchmove', function(e) {
                            var x = e.pageX || e.originalEvent.touches[0].pageX;
                            var walked = (x - startX) * 2;
                            $this.scrollLeft(scrollLeft - walked);
                        });
    
                        $(document).on('mouseup touchend', function() {
                            $(document).off('mousemove touchmove mouseup touchend');
                        });
                    });
                })(jQuery);
            </script>
            <?php
        endif;
    }
}
