<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class Skill_Featured_Projects_Widget extends \Elementor\Widget_Base {

    public function get_name() {
        return 'skill_featured_projects';
    }

    public function get_title() {
        return __( 'پروژه‌های منتخب', 'skill' );
    }

    public function get_icon() {
        return 'eicon-gallery-grid';
    }

    public function get_categories() {
        return [ 'skill-category' ];
    }

    protected function register_controls() {

        // === Content Section ===
        $this->start_controls_section(
            'content_section',
            [
                'label' => __( 'تنظیمات محتوا', 'skill' ),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        // Select projects
        $this->add_control(
            'selected_projects',
            [
                'label'       => __( 'انتخاب پروژه‌ها', 'skill' ),
                'type'        => \Elementor\Controls_Manager::SELECT2,
                'multiple'    => true,
                'options'     => $this->get_projects_list(),
                'label_block' => true,
            ]
        );

        // Order options (applied among selected items)
        $this->add_control(
            'order_by',
            [
                'label'   => __( 'مرتب‌سازی بر اساس', 'skill' ),
                'type'    => \Elementor\Controls_Manager::SELECT,
                'default' => 'selected',
                'options' => [
                    'selected' => __( 'ترتیب انتخاب', 'skill' ),
                    'date'     => __( 'تاریخ', 'skill' ),
                    'title'    => __( 'عنوان', 'skill' ),
                    'rand'     => __( 'تصادفی', 'skill' ),
                ],
            ]
        );

        $this->add_control(
            'order',
            [
                'label'     => __( 'ترتیب', 'skill' ),
                'type'      => \Elementor\Controls_Manager::SELECT,
                'default'   => 'DESC',
                'options'   => [
                    'DESC' => __( 'نزولی', 'skill' ),
                    'ASC'  => __( 'صعودی', 'skill' ),
                ],
                'condition' => [ 'order_by!' => 'selected' ],
            ]
        );

        $this->add_control(
            'max_items',
            [
                'label'       => __( 'حداکثر تعداد نمایش', 'skill' ),
                'type'        => \Elementor\Controls_Manager::NUMBER,
                'min'         => 1,
                'step'        => 1,
                'description' => __( 'برای نامحدود خالی بگذارید', 'skill' ),
            ]
        );

        // Toggles
        $this->add_control(
            'show_image',
            [
                'label'   => __( 'نمایش تصویر شاخص', 'skill' ),
                'type'    => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_title',
            [
                'label'   => __( 'نمایش عنوان', 'skill' ),
                'type'    => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_excerpt',
            [
                'label' => __( 'نمایش خلاصه/توضیح کوتاه', 'skill' ),
                'type'  => \Elementor\Controls_Manager::SWITCHER,
            ]
        );

        $this->add_control(
            'show_button',
            [
                'label'   => __( 'نمایش دکمه', 'skill' ),
                'type'    => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'button_text',
            [
                'label'     => __( 'متن دکمه', 'skill' ),
                'type'      => \Elementor\Controls_Manager::TEXT,
                'default'   => __( 'مشاهده پروژه', 'skill' ),
                'dynamic'   => [ 'active' => true ],
                'condition' => [ 'show_button' => 'yes' ],
            ]
        );

        $this->add_control(
            'new_tab',
            [
                'label'        => __( 'باز شدن لینک در تب جدید', 'skill' ),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'return_value' => 'yes',
                'condition'    => [ 'show_button' => 'yes' ],
            ]
        );

        // Image size
        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name'    => 'thumb',
                'default' => 'medium',
                'exclude' => [ 'custom' ],
                'condition' => [ 'show_image' => 'yes' ],
            ]
        );

        $this->end_controls_section();

        // === Layout Section (Content Tab) ===
        $this->start_controls_section(
            'layout_section',
            [
                'label' => __( 'چیدمان', 'skill' ),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_responsive_control(
            'columns',
            [
                'label'   => __( 'ستون‌ها', 'skill' ),
                'type'    => \Elementor\Controls_Manager::SELECT,
                'default' => '3',
                'options' => [
                    '1' => '1',
                    '2' => '2',
                    '3' => '3',
                    '4' => '4',
                    '5' => '5',
                    '6' => '6',
                ],
                'selectors' => [
                    '{{WRAPPER}} .skill-featured-projects' => 'display:grid;grid-template-columns:repeat({{VALUE}}, minmax(0, 1fr));',
                ],
            ]
        );

        $this->add_responsive_control(
            'gap',
            [
                'label' => __( 'فاصله کارت‌ها', 'skill' ),
                'type'  => \Elementor\Controls_Manager::SLIDER,
                'range' => [ 'px' => [ 'min' => 0, 'max' => 60 ] ],
                'default' => [ 'size' => 20, 'unit' => 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .skill-featured-projects' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // === Card Style ===
        $this->start_controls_section(
            'card_style_section',
            [
                'label' => __( 'استایل کارت', 'skill' ),
                'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        // Background color
        $this->add_control(
            'box_bg_color',
            [
                'label' => __( 'رنگ پس‌زمینه کارت', 'skill' ),
                'type'  => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill-project-card' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'card_padding',
            [
                'label' => __( 'فاصله داخلی کارت', 'skill' ),
                'type'  => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .skill-project-card' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'card_radius',
            [
                'label' => __( 'گردی گوشه‌ها', 'skill' ),
                'type'  => \Elementor\Controls_Manager::SLIDER,
                'range' => [ 'px' => [ 'min' => 0, 'max' => 40 ] ],
                'selectors' => [
                    '{{WRAPPER}} .skill-project-card' => 'border-radius: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .skill-project-thumb img' => 'border-radius: calc({{SIZE}}{{UNIT}} - 5px);',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'card_shadow',
                'selector' => '{{WRAPPER}} .skill-project-card',
            ]
        );

        $this->end_controls_section();

        // === Title Style ===
        $this->start_controls_section(
            'title_style_section',
            [
                'label' => __( 'استایل عنوان', 'skill' ),
                'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [ 'show_title' => 'yes' ],
            ]
        );

        // Title color
        $this->add_control(
            'title_color',
            [
                'label' => __( 'رنگ عنوان', 'skill' ),
                'type'  => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .skill-project-title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .skill-project-title',
            ]
        );

        $this->end_controls_section();

        // سبک دکمه از کلاس سراسری .btn-more تبعیت می‌کند؛ کنترل‌های استایل دکمه حذف شدند
    }

    /**
     * Get list of projects (CPT: projects)
     */
    private function get_projects_list() {
        $projects = get_posts( [
            'post_type' => 'project',
            'posts_per_page' => -1
        ] );
        $options = [];
        foreach ( $projects as $project ) {
            $options[ $project->ID ] = $project->post_title;
        }
        return $options;
    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        if ( empty( $settings['selected_projects'] ) || ! is_array( $settings['selected_projects'] ) ) {
            echo '<p>' . esc_html__( 'هیچ پروژه‌ای انتخاب نشده است.', 'skill' ) . '</p>';
            return;
        }

        $selected_ids = array_map( 'intval', $settings['selected_projects'] );
        $selected_ids = array_filter( $selected_ids );

        // Apply ordering among selected items if needed
        $projects = [];
        $order_by = isset( $settings['order_by'] ) ? $settings['order_by'] : 'selected';
        $order    = isset( $settings['order'] ) ? $settings['order'] : 'DESC';
        $max      = ! empty( $settings['max_items'] ) ? intval( $settings['max_items'] ) : 0;

        if ( $order_by !== 'selected' ) {
            $query_args = [
                'post_type'      => 'project',
                'post__in'       => $selected_ids,
                'posts_per_page' => $max > 0 ? $max : -1,
                'orderby'        => $order_by,
                'order'          => $order,
                'no_found_rows'  => true,
            ];
            if ( $order_by === 'rand' ) {
                unset( $query_args['order'] );
            }
            $projects = get_posts( $query_args );
        } else {
            // Keep selected order
            if ( $max > 0 ) {
                $selected_ids = array_slice( $selected_ids, 0, $max );
            }
            foreach ( $selected_ids as $pid ) {
                $post_obj = get_post( $pid );
                if ( $post_obj && $post_obj->post_type === 'project' ) {
                    $projects[] = $post_obj;
                }
            }
        }

        if ( empty( $projects ) ) {
            echo '<p>' . esc_html__( 'پروژه‌ای برای نمایش وجود ندارد.', 'skill' ) . '</p>';
            return;
        }

        $image_size = ! empty( $settings['thumb_size'] ) ? $settings['thumb_size'] : 'medium';
        $open_new_tab = ! empty( $settings['new_tab'] ) && $settings['new_tab'] === 'yes';
        $rel_attr = $open_new_tab ? ' rel="noopener noreferrer"' : '';
        $target_attr = $open_new_tab ? ' target="_blank"' : '';

        // Per-instance base styles to ensure grid (also handled by selectors from controls)
        $uid = $this->get_id();
        $columns_desktop = ! empty( $settings['columns'] ) ? intval( $settings['columns'] ) : 3;
        $columns_tablet  = ! empty( $settings['columns_tablet'] ) ? intval( $settings['columns_tablet'] ) : 2;
        $columns_mobile  = ! empty( $settings['columns_mobile'] ) ? intval( $settings['columns_mobile'] ) : 1;
        $gap_desktop     = isset( $settings['gap']['size'] ) ? $settings['gap']['size'] : 20;
        $gap_tablet      = isset( $settings['gap_tablet']['size'] ) ? $settings['gap_tablet']['size'] : $gap_desktop;
        $gap_mobile      = isset( $settings['gap_mobile']['size'] ) ? $settings['gap_mobile']['size'] : $gap_tablet;

        echo '<style>
.elementor-element-' . esc_attr( $uid ) . ' .skill-featured-projects{display:grid;grid-template-columns:repeat(' . intval( $columns_desktop ) . ',minmax(0,1fr));gap:' . esc_attr( $gap_desktop ) . 'px;}
@media(max-width:1024px){.elementor-element-' . esc_attr( $uid ) . ' .skill-featured-projects{grid-template-columns:repeat(' . intval( $columns_tablet ) . ',minmax(0,1fr));gap:' . esc_attr( $gap_tablet ) . 'px;}}
@media(max-width:767px){.elementor-element-' . esc_attr( $uid ) . ' .skill-featured-projects{grid-template-columns:repeat(' . intval( $columns_mobile ) . ',minmax(0,1fr));gap:' . esc_attr( $gap_mobile ) . 'px;}}
.elementor-element-' . esc_attr( $uid ) . ' .skill-project-card{
  position:relative;
  border-radius:22px;
  overflow:hidden;
  background:#0b1424;
  box-shadow:0 6px 24px rgba(0,0,0,0.12);
}
.elementor-element-' . esc_attr( $uid ) . ' .skill-project-thumb img{
  width:100%;
  height:220px;
  display:block;
  object-fit:cover;
}
.elementor-element-' . esc_attr( $uid ) . ' .skill-project-overlay{
  position:absolute;inset:0;display:flex;flex-direction:column;justify-content:flex-end;gap:10px;
  padding:16px 16px 14px 16px;color:#fff;
  background:linear-gradient(180deg, rgba(0,0,0,0) 35%, rgba(0,0,0,0.72) 100%);
  opacity:0;transform:translateY(12px);
  transition:opacity .28s ease, transform .28s ease;
}
.elementor-element-' . esc_attr( $uid ) . ' .skill-project-card:hover .skill-project-overlay{opacity:1;transform:translateY(0);background:#031d53b5;} 
.elementor-element-' . esc_attr( $uid ) . ' .skill-project-title{margin:0 0 4px 0;font-size:1.05rem;line-height:1.6;color:#fff;}
.elementor-element-' . esc_attr( $uid ) . ' .skill-project-title:before{content:"";display:inline-block;width:8px;height:8px;background:#ffffff;border-radius:2px;margin-inline-end:8px;vertical-align:middle;}
.elementor-element-' . esc_attr( $uid ) . ' .skill-project-excerpt{margin:0 0 8px 0;color:#e6efff;opacity:.95;font-size:.95rem;line-height:1.8;display:-webkit-box;-webkit-line-clamp:3;-webkit-box-orient:vertical;overflow:hidden;}
.elementor-element-' . esc_attr( $uid ) . ' .skill-project-link.btn-more{display:inline-flex !important;width:auto !important;max-width:100%;align-self:flex-start;padding:5px 14px 5px 3px;}
.elementor-element-' . esc_attr( $uid ) . ' .skill-project-link:not(.btn-more){align-self:flex-start;display:inline-flex;align-items:center;gap:8px;padding:8px 16px;border-radius:20px;background:#ffffff;color:#0b1424;text-decoration:none;font-size:.92rem;font-weight:600;transition:transform .2s ease, box-shadow .2s ease;box-shadow:0 2px 8px rgba(0,0,0,0.15);} 
.elementor-element-' . esc_attr( $uid ) . ' .skill-project-link:not(.btn-more):hover{transform:translateY(-2px);box-shadow:0 6px 18px rgba(0,0,0,0.28);} 
</style>';

        echo '<div class="skill-featured-projects">';
        foreach ( $projects as $project ) {
            $project_id = $project->ID;
            $title = get_the_title( $project_id );
            $link  = get_permalink( $project_id );
            $short = get_post_meta( $project_id, '_project_description_short', true );
            if ( empty( $short ) ) {
                $short = get_the_excerpt( $project_id );
            }

            echo '<div class="skill-project-card">';

            if ( ! empty( $settings['show_image'] ) && $settings['show_image'] === 'yes' ) {
                $thumb_html = get_the_post_thumbnail( $project_id, $image_size, [ 'loading' => 'lazy' ] );
                if ( $thumb_html ) {
                    echo '<div class="skill-project-thumb">' . $thumb_html . '</div>';
                }
            }

            echo '<div class="skill-project-overlay">';
            if ( ! empty( $settings['show_title'] ) && $settings['show_title'] === 'yes' ) {
                echo '<h3 class="skill-project-title">' . esc_html( $title ) . '</h3>';
            }
            if ( ! empty( $settings['show_excerpt'] ) && $settings['show_excerpt'] === 'yes' && ! empty( $short ) ) {
                echo '<div class="skill-project-excerpt">' . esc_html( $short ) . '</div>';
            }
            if ( ! empty( $settings['show_button'] ) && $settings['show_button'] === 'yes' ) {
                $btn_text = ! empty( $settings['button_text'] ) ? $settings['button_text'] : __( 'مشاهده پروژه', 'skill' );
                echo '<a class="skill-project-link btn-more" href="' . esc_url( $link ) . '"' . $target_attr . $rel_attr . '>'
                    . '<span class="btn-more-text">' . esc_html( $btn_text ) . '</span>'
                    . '<span class="btn-more-icon"><i class="fa-solid fa-arrow-left"></i></span>'
                    . '</a>';
            }
            echo '</div>';

            echo '</div>';
        }
        echo '</div>';
    }

    protected function _content_template() {
        ?>
        <#
        if ( settings.selected_projects.length === 0 ) {
            print( '<?php echo __( "هیچ پروژه‌ای انتخاب نشده است.", "skill" ); ?>' );
        } else {
        #>
        <div class="skill-featured-projects">
            <# _.each( settings.selected_projects, function( project_id ) { #>
                <div class="skill-project-card">
                    <# if ( settings.show_title === 'yes' ) { #>
                        <h3 class="skill-project-title"><?php echo esc_html__( 'پروژه', 'skill' ); ?> {{ project_id }}</h3>
                    <# } #>
                    <# if ( settings.show_button === 'yes' ) { #>
                        <a class="skill-project-link" href="#">{{ settings.button_text || '<?php echo esc_html__( 'مشاهده پروژه', 'skill' ); ?>' }}</a>
                    <# } #>
                </div>
            <# }); #>
        </div>
        <# } #>
        <?php
    }
}

