<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class skill_project_gallery_Widget extends \Elementor\Widget_Base {
    public function get_name() {
        return 'skill_project_gallery_Widget';
    }
    public function get_title() {
        return esc_html__('گالری تصاویر پروژه', 'skill');
    }
    public function get_icon() {
        return 'eicon-gallery-grid';
    }
    public function get_categories() {
        return ['skill-category'];
    }
    
    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('تنظیمات گالری', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'gallery_title',
            [
                'label' => esc_html__('عنوان گالری', 'skill'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('گالری تصاویر پروژه', 'skill'),
                'placeholder' => esc_html__('عنوان گالری را وارد کنید', 'skill'),
            ]
        );

        $this->add_control(
            'gallery_images',
            [
                'label' => esc_html__('تصاویر گالری', 'skill'),
                'type' => \Elementor\Controls_Manager::GALLERY,
                'default' => [],
                'description' => esc_html__('تصاویر گالری پروژه را انتخاب کنید', 'skill'),
            ]
        );

        $this->add_control(
            'columns',
            [
                'label' => esc_html__('تعداد ستون‌ها', 'skill'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '3',
                'options' => [
                    '1' => esc_html__('1 ستون', 'skill'),
                    '2' => esc_html__('2 ستون', 'skill'),
                    '3' => esc_html__('3 ستون', 'skill'),
                    '4' => esc_html__('4 ستون', 'skill'),
                ],
            ]
        );

        $this->add_control(
            'lightbox',
            [
                'label' => esc_html__('نمایش در لایت باکس', 'skill'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('فعال', 'skill'),
                'label_off' => esc_html__('غیرفعال', 'skill'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

        // Style Tab
        $this->start_controls_section(
            'style_section',
            [
                'label' => esc_html__('استایل', 'skill'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('رنگ عنوان', 'skill'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .project-gallery-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .project-gallery-title',
            ]
        );

        $this->add_control(
            'image_border_radius',
            [
                'label' => esc_html__('گردی گوشه تصاویر', 'skill'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 10,
                ],
                'selectors' => [
                    '{{WRAPPER}} .gallery-item img' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        
        // Get gallery images from post meta if not set in Elementor
        $gallery_images = $settings['gallery_images'];
        if (empty($gallery_images)) {
            $post_id = get_the_ID();
            $meta_gallery = get_post_meta($post_id, '_project_gallery', true);
            
            if (is_array($meta_gallery) && !empty($meta_gallery)) {
                $gallery_images = array();
                foreach ($meta_gallery as $image_id) {
                    $image_url = wp_get_attachment_image_url($image_id, 'medium');
                    if ($image_url) {
                        $gallery_images[] = array(
                            'id' => $image_id,
                            'url' => $image_url
                        );
                    }
                }
            }
        }

        if (empty($gallery_images)) {
            return;
        }

        $columns = $settings['columns'];
        $lightbox = $settings['lightbox'] === 'yes';
        ?>
        <div class="skill-project-gallery-widget">
            <?php if ($settings['gallery_title']): ?>
                <h3 class="project-gallery-title"><?php echo esc_html($settings['gallery_title']); ?></h3>
            <?php endif; ?>
            
            <div class="gallery-grid gallery-columns-<?php echo esc_attr($columns); ?>">
                <?php foreach ($gallery_images as $image): 
                    $image_id = $image['id'];
                    $image_url = $image['url'];
                    $image_full = wp_get_attachment_image_url($image_id, 'full');
                    $image_alt = get_post_meta($image_id, '_wp_attachment_image_alt', true);
                ?>
                    <div class="gallery-item" data-full-image="<?php echo esc_url($image_full); ?>">
                        <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($image_alt); ?>">
                    </div>
                <?php endforeach; ?>
            </div>
        </div>

        <style>
        .skill-project-gallery-widget {
            margin: 2rem 0;
        }
        
        .project-gallery-title {
            margin-bottom: 1.5rem;
            font-size: 1.5rem;
            font-weight: 600;
            color: #333;
        }
        
        .gallery-grid {
            display: grid;
            gap: 1rem;
        }
        
        .gallery-columns-1 { grid-template-columns: 1fr; }
        .gallery-columns-2 { grid-template-columns: repeat(2, 1fr); }
        .gallery-columns-3 { grid-template-columns: repeat(3, 1fr); }
        .gallery-columns-4 { grid-template-columns: repeat(4, 1fr); }
        
        .gallery-item {
            position: relative;
            overflow: hidden;
            border-radius: 10px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .gallery-item:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }
        
        .gallery-item img {
            width: 100%;
            height: 200px;
            object-fit: cover;
            display: block;
            transition: transform 0.3s ease;
        }
        
        .gallery-item:hover img {
            transform: scale(1.05);
        }
        
        /* Apply same styles to regular project galleries */
        .project-gallery .gallery-item {
            position: relative;
            overflow: hidden;
            border-radius: 10px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .project-gallery .gallery-item:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }
        
        .project-gallery .gallery-item img {
            width: 100%;
            height: 200px;
            object-fit: cover;
            display: block;
            transition: transform 0.3s ease;
        }
        
        .project-gallery .gallery-item:hover img {
            transform: scale(1.05);
        }
        
        /* Gallery Modal Styles */
        .skill-gallery-modal {
            display: none;
            position: fixed;
            z-index: 9999;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.95);
            backdrop-filter: blur(10px);
        }
        
        .skill-gallery-modal.active {
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .skill-gallery-content {
            position: relative;
            max-width: 90%;
            max-height: 90%;
            text-align: center;
        }
        
        .skill-gallery-content img {
            max-width: 100%;
            max-height: 80vh;
            object-fit: contain;
            border-radius: 10px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        }
        
        .skill-gallery-close {
            position: absolute;
            top: -50px;
            right: 0;
            color: #fff;
            font-size: 40px;
            font-weight: bold;
            cursor: pointer;
            z-index: 10000;
            background: rgba(0, 0, 0, 0.5);
            width: 50px;
            height: 50px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
        }
        
        .skill-gallery-close:hover {
            background: rgba(255, 255, 255, 0.2);
            transform: scale(1.1);
        }
        
        .skill-gallery-nav {
            position: absolute;
            top: 50%;
            transform: translateY(-50%);
            background: rgba(255, 255, 255, 0.2);
            color: #fff;
            border: none;
            padding: 15px 20px;
            cursor: pointer;
            border-radius: 50%;
            font-size: 24px;
            transition: all 0.3s ease;
            backdrop-filter: blur(10px);
        }
        
        .skill-gallery-nav:hover {
            background: rgba(255, 255, 255, 0.3);
            transform: scale(1.1);
        }
        
        .skill-gallery-prev {
            left: 20px;
        }
        
        .skill-gallery-next {
            right: 20px;
        }
        
        .skill-gallery-counter {
            position: absolute;
            bottom: -40px;
            left: 50%;
            transform: translateX(-50%);
            color: #fff;
            font-size: 16px;
            background: rgba(0, 0, 0, 0.5);
            padding: 10px 20px;
            border-radius: 20px;
            backdrop-filter: blur(10px);
        }
        
        @media (max-width: 768px) {
            .gallery-columns-2,
            .gallery-columns-3,
            .gallery-columns-4 {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .skill-gallery-nav {
                padding: 10px 15px;
                font-size: 18px;
            }
            
            .skill-gallery-prev {
                left: 10px;
            }
            
            .skill-gallery-next {
                right: 10px;
            }
        }
        
        @media (max-width: 480px) {
            .gallery-columns-2,
            .gallery-columns-3,
            .gallery-columns-4 {
                grid-template-columns: 1fr;
            }
        }
        </style>
        
        <script>
        (function() {
            // Create modal only once
            if (document.querySelector('.skill-gallery-modal')) {
                return;
            }
            
            // Create modal elements
            var modal = document.createElement('div');
            modal.className = 'skill-gallery-modal';
            modal.innerHTML = `
                <div class="skill-gallery-content">
                    <span class="skill-gallery-close">&times;</span>
                    <img src="" alt="">
                    <button class="skill-gallery-nav skill-gallery-next">‹</button>
                    <button class="skill-gallery-nav skill-gallery-prev">›</button>
                    <div class="skill-gallery-counter"></div>
                </div>
            `;
            document.body.appendChild(modal);
            
            var currentIndex = 0;
            var galleryImages = [];
            
            // Initialize gallery
            function initGallery() {
                // Select all gallery items from both Elementor widget and regular project galleries
                var galleryItems = document.querySelectorAll('.skill-project-gallery-widget .gallery-item, .project-gallery .gallery-item');
                galleryImages = [];
                
                galleryItems.forEach(function(item, index) {
                    var img = item.querySelector('img');
                    if (img) {
                        // Get full size image URL from data attribute or link href
                        var fullSizeUrl = item.getAttribute('data-full-image');
                        
                        // If no data attribute, try to get from link href
                        if (!fullSizeUrl) {
                            var link = item.querySelector('a');
                            if (link) {
                                fullSizeUrl = link.getAttribute('href');
                            }
                        }
                        
                        // Fallback to image src if no full size URL found
                        if (!fullSizeUrl) {
                            fullSizeUrl = img.src;
                        }
                        
                        galleryImages.push({
                            src: fullSizeUrl,
                            alt: img.alt
                        });
                        
                        // Add click event
                        item.addEventListener('click', function(e) {
                            e.preventDefault();
                            openGallery(index);
                        });
                    }
                });
            }
            
            // Open gallery
            function openGallery(index) {
                currentIndex = index;
                updateGalleryImage();
                modal.classList.add('active');
                document.body.style.overflow = 'hidden';
            }
            
            // Update gallery image
            function updateGalleryImage() {
                if (galleryImages[currentIndex]) {
                    var img = modal.querySelector('img');
                    var counter = modal.querySelector('.skill-gallery-counter');
                    
                    img.src = galleryImages[currentIndex].src;
                    img.alt = galleryImages[currentIndex].alt;
                    counter.textContent = (currentIndex + 1) + ' / ' + galleryImages.length;
                }
            }
            
            // Navigation
            function nextImage() {
                currentIndex = (currentIndex + 1) % galleryImages.length;
                updateGalleryImage();
            }
            
            function prevImage() {
                currentIndex = (currentIndex - 1 + galleryImages.length) % galleryImages.length;
                updateGalleryImage();
            }
            
            // Close gallery
            function closeGallery() {
                modal.classList.remove('active');
                document.body.style.overflow = 'auto';
            }
            
            // Event listeners
            modal.querySelector('.skill-gallery-close').addEventListener('click', closeGallery);
            modal.querySelector('.skill-gallery-next').addEventListener('click', nextImage);
            modal.querySelector('.skill-gallery-prev').addEventListener('click', prevImage);
            
            // Close on background click
            modal.addEventListener('click', function(e) {
                if (e.target === modal) {
                    closeGallery();
                }
            });
            
            // Keyboard navigation
            document.addEventListener('keydown', function(e) {
                if (modal.classList.contains('active')) {
                    if (e.key === 'Escape') {
                        closeGallery();
                    } else if (e.key === 'ArrowRight') {
                        nextImage();
                    } else if (e.key === 'ArrowLeft') {
                        prevImage();
                    }
                }
            });
            
            // Initialize on DOM ready
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', initGallery);
            } else {
                initGallery();
            }
            
            // Initialize for Elementor frontend
            if (window.elementorFrontend) {
                window.elementorFrontend.hooks.addAction('frontend/element_ready/widget', function($scope) {
                    if ($scope.find('.skill-project-gallery-widget').length) {
                        setTimeout(initGallery, 100);
                    }
                });
            }
        })();
        </script>
        

        <?php
    }
}
