<?php
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Elementor custom progress bar widget with 4 rounded squares
class skill_progress_bar_e_Widget extends Widget_Base {
    public function get_name() {
        return 'skill-progressbar-e';
    }

    public function get_title() {
        // Widget title in Elementor panel
        return __('نوار پیشرفت مربعی', 'skill');
    }

    public function get_icon() {
        return 'eicon-skill-bar';
    }

    public function get_categories() {
        // Custom Elementor category
        return [ 'skill-category' ];
    }

    protected function _register_controls() {
        // Widget Title Controls
        $this->start_controls_section(
            'section_widget_title',
            [
                'label' => __('عنوان ویجت', 'skill'),
            ]
        );
        // Title+Desc controls like skill-title-desc-widget
        $this->add_control('show_title', [
            'label' => __('نمایش عنوان', 'skill'),
            'type' => Controls_Manager::SWITCHER,
            'label_on' => __('بله', 'skill'),
            'label_off' => __('خیر', 'skill'),
            'return_value' => 'yes',
            'default' => 'yes',
        ]);
        $this->add_control('title_text', [
            'label' => __('عنوان', 'skill'),
            'type' => Controls_Manager::TEXT,
            'default' => __('مهارت های من', 'skill'),
            'condition' => [ 'show_title' => 'yes' ],
            'label_block' => true,
        ]);
        $this->add_control('show_dot', [
            'label' => __('نمایش مربع آبی', 'skill'),
            'type' => Controls_Manager::SWITCHER,
            'label_on' => __('بله', 'skill'),
            'label_off' => __('خیر', 'skill'),
            'return_value' => 'yes',
            'default' => 'yes',
        ]);
        $this->add_control('dot_position', [
            'label' => __('جایگاه مربع', 'skill'),
            'type' => Controls_Manager::SELECT,
            'options' => [ 'right' => __('راست عنوان', 'skill'), 'left' => __('چپ عنوان', 'skill') ],
            'default' => 'right',
            'condition' => [ 'show_dot' => 'yes' ],
        ]);
        $this->add_control('show_desc', [
            'label' => __('نمایش توضیح کوتاه', 'skill'),
            'type' => Controls_Manager::SWITCHER,
            'label_on' => __('بله', 'skill'),
            'label_off' => __('خیر', 'skill'),
            'return_value' => 'yes',
            'default' => '',
        ]);
        $this->add_control('desc_text', [
            'label' => __('توضیح کوتاه', 'skill'),
            'type' => Controls_Manager::TEXTAREA,
            'rows' => 2,
            'condition' => [ 'show_desc' => 'yes' ],
        ]);
        // Keep legacy controls for backward-compat (not used in render)
        $this->add_control('widget_title', [ 'label' => __('(قدیمی) عنوان ویجت', 'skill'), 'type' => Controls_Manager::TEXT, 'default' => __('مهارت های من', 'skill'), 'separator' => 'before' ]);
        $this->end_controls_section();

        // Modern Design Controls
        $this->start_controls_section(
            'section_design',
            [
                'label' => __('تنظیمات طراحی مدرن', 'skill'),
            ]
        );
        $this->add_control(
            'design_style',
            [
                'label' => __('سبک طراحی', 'skill'),
                'type' => Controls_Manager::SELECT,
                'default' => 'glassmorphism',
                'options' => [
                    'glassmorphism' => __('شیشه‌ای مدرن', 'skill'),
                    'gradient' => __('گرادیانت', 'skill'),
                    'neumorphism' => __('نئومورفیک', 'skill'),
                    'minimal' => __('مینیمال', 'skill'),
                ],
            ]
        );
        $this->add_control(
            'enable_animations',
            [
                'label' => __('انیمیشن‌های مدرن', 'skill'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );
        $this->end_controls_section();

        // Title styles like skill-title-desc-widget
        $this->start_controls_section(
            'tdw_style',
            [ 'label' => __('استایل عنوان/توضیح', 'skill') , ]
        );
        $this->add_responsive_control('tdw_align', [
            'label' => __('چینش', 'skill'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'right' => ['title' => __('راست', 'skill'), 'icon' => 'eicon-text-align-right'],
                'center' => ['title' => __('وسط', 'skill'), 'icon' => 'eicon-text-align-center'],
                'left' => ['title' => __('چپ', 'skill'), 'icon' => 'eicon-text-align-left'],
            ],
            'default' => 'center',
            'selectors' => [ '{{WRAPPER}} .tdw-wrap' => 'text-align: {{VALUE}};' ],
        ]);
        $this->add_control('tdw_title_color', [
            'label' => __('رنگ عنوان', 'skill'),
            'type' => Controls_Manager::COLOR,
            'default' => '#1a1a1a',
            'selectors' => [ '{{WRAPPER}} .tdw-title' => 'color: {{VALUE}};' ],
        ]);
        $this->add_group_control(\Elementor\Group_Control_Typography::get_type(), [ 'name' => 'tdw_title_typo', 'selector' => '{{WRAPPER}} .tdw-title' ]);
        $this->add_control('tdw_desc_color', [
            'label' => __('رنگ توضیح', 'skill'),
            'type' => Controls_Manager::COLOR,
            'default' => 'rgba(0,0,0,0.6)',
            'selectors' => [ '{{WRAPPER}} .tdw-desc' => 'color: {{VALUE}};' ],
        ]);
        $this->add_group_control(\Elementor\Group_Control_Typography::get_type(), [ 'name' => 'tdw_desc_typo', 'selector' => '{{WRAPPER}} .tdw-desc' ]);
        $this->add_control('tdw_dot_color', [ 'label' => __('رنگ مربع', 'skill'), 'type' => Controls_Manager::COLOR, 'default' => '#1e73be', 'selectors' => [ '{{WRAPPER}} .tdw-dot' => 'background-color: {{VALUE}};' ]]);
        $this->add_control('tdw_dot_size', [ 'label' => __('سایز مربع (px)', 'skill'), 'type' => Controls_Manager::SLIDER, 'size_units' => ['px'], 'range' => [ 'px' => [ 'min' => 6, 'max' => 24 ] ], 'default' => ['unit' => 'px', 'size' => 10], 'selectors' => [ '{{WRAPPER}} .tdw-dot' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};' ]]);
        $this->end_controls_section();

        // Repeater for progress items
        $this->start_controls_section(
            'section_content',
            [
                'label' => __('موارد نوار پیشرفت', 'skill'),
            ]
        );
        $repeater = new Repeater();
        $repeater->add_control(
            'title',
            [
                'label' => __('عنوان مهارت', 'skill'),
                'type' => Controls_Manager::TEXT,
                'default' => __('مهارت نمونه', 'skill'),
            ]
        );
        $repeater->add_control(
            'percent',
            [
                'label' => __('درصد', 'skill'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ '' ],
                'range' => [
                    '' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 25,
                    ],
                ],
                'default' => [ 'size' => 50 ],
            ]
        );
        $repeater->add_control(
            'main_color',
            [
                'label' => __('رنگ اصلی', 'skill'),
                'type' => Controls_Manager::COLOR,
                'default' => '#6366f1',
            ]
        );
        $repeater->add_control(
            'blur_color',
            [
                'label' => __('رنگ پس‌زمینه', 'skill'),
                'type' => Controls_Manager::COLOR,
                'default' => 'rgba(99, 102, 241, 0.1)',
            ]
        );
        $repeater->add_control(
            'percent_color',
            [
                'label' => __('رنگ درصد', 'skill'),
                'type' => Controls_Manager::COLOR,
                'default' => '#1a1a1a',
            ]
        );
        $this->add_control(
            'items',
            [
                'label' => __('موارد', 'skill'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'title' => __('مهارت نمونه', 'skill'),
                        'percent' => [ 'size' => 50 ],
                        'main_color' => '#6366f1',
                        'blur_color' => 'rgba(99, 102, 241, 0.1)',
                    ],
                ],
                'title_field' => '{{{ title }}}',
            ]
        );
        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $design_style = $settings['design_style'] ?? 'glassmorphism';
        $enable_animations = $settings['enable_animations'] === 'yes';
        ?>
        <div class="skill-progressbar-e-widget skill-<?php echo esc_attr($design_style); ?>" data-animations="<?php echo $enable_animations ? 'enabled' : 'disabled'; ?>">
            <div class="tdw-wrap" dir="rtl" style="margin-bottom:22px;">
                <div class="tdw-title-row" style="display:inline-flex; align-items:center; gap:8px; <?php echo (($settings['dot_position'] ?? 'right') === 'right') ? 'flex-direction: row-reverse;' : ''; ?>">
                    <?php if (!empty($settings['show_title']) && $settings['show_title'] === 'yes' && !empty($settings['title_text'])): ?>
                        <h2 class="tdw-title"><?php echo esc_html($settings['title_text']); ?></h2>
                    <?php endif; ?>
                    <?php if (!empty($settings['show_dot']) && $settings['show_dot'] === 'yes'): ?>
                        <span class="tdw-dot" aria-hidden="true" style="display:inline-block; border-radius:3px;"></span>
                    <?php endif; ?>
                </div>
                <?php if (!empty($settings['show_desc']) && $settings['show_desc'] === 'yes' && !empty($settings['desc_text'])): ?>
                    <div class="tdw-desc" style="margin-top:10px;">
                        <?php echo esc_html($settings['desc_text']); ?>
                    </div>
                <?php endif; ?>
            </div>
            <?php if (!empty($settings['items'])): ?>
                <div class="skill-progressbar-e-items">
                <?php foreach ($settings['items'] as $index => $item):
                    if (isset($item['percent']['size'])) {
                        $percent = intval($item['percent']['size']);
                    } elseif (is_numeric($item['percent'])) {
                        $percent = intval($item['percent']);
                    } else {
                        $percent = 0;
                    }
                    $main_color = $item['main_color'];
                    $blur_color = $item['blur_color'];
                    $percent_color = isset($item['percent_color']) ? $item['percent_color'] : $main_color;
                    $filled_squares = floor($percent / 25); // Calculate how many squares should be filled
                ?>
                <div class="skill-progressbar-e-item" data-index="<?php echo $index; ?>">
                    <div class="skill-progressbar-e-content">
                        <div class="skill-progressbar-e-title"><?php echo esc_html($item['title']); ?></div>
                        <div class="skill-progressbar-e-percent"><?php echo esc_html($percent); ?>%</div>
                    </div>
                    
                    <div class="skill-progressbar-e-squares">
                        <?php for ($i = 1; $i <= 4; $i++) : ?>
                            <div class="skill-progressbar-e-square">
                                <div class="skill-progressbar-e-square-fill" style="width: <?php echo $i <= $filled_squares ? '100%' : '0%'; ?>"></div>
                                <div class="skill-progressbar-e-square-glow"></div>
                            </div>
                        <?php endfor; ?>
                    </div>
                </div>
                <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
        <style>
        .skill-progressbar-e-widget {
            display: block;
            width: 100%;
        }
        
        .skill-progressbar-e-items {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
            gap: 24px;
        }
        
        /* Glassmorphism Style - 2025 Modern */
        .skill-glassmorphism .skill-progressbar-e-item {
            background: rgba(255, 255, 255, 0.08);
            backdrop-filter: blur(20px);
            -webkit-backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 24px;
            padding: 28px;
            box-shadow: 
                0 8px 32px rgba(0, 0, 0, 0.1),
                inset 0 1px 0 rgba(255, 255, 255, 0.2);
            position: relative;
            overflow: hidden;
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            display: block;
        }
        
        .skill-glassmorphism .skill-progressbar-e-item::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.1), transparent);
            transition: left 0.8s ease;
            z-index: 1;
            pointer-events: none;
        }
        
        .skill-glassmorphism .skill-progressbar-e-item:hover::before {
            left: 100%;
        }
        
        .skill-glassmorphism .skill-progressbar-e-item:hover {
            transform: translateY(-8px) scale(1.02);
            box-shadow: 
                0 20px 40px rgba(0, 0, 0, 0.15),
                inset 0 1px 0 rgba(255, 255, 255, 0.3);
        }

        .skill-progressbar-e-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        /* Ensure content stays above shimmer */
        .skill-glassmorphism .skill-progressbar-e-content {
            position: relative;
            z-index: 2;
        }
        
        /* Ensure squares stay visible during shimmer */
        .skill-glassmorphism .skill-progressbar-e-square {
            background: rgba(255, 255, 255, 0.2) !important;
            border: 1px solid rgba(255, 255, 255, 0.3) !important;
            position: relative;
            z-index: 3;
        }
        
        .skill-glassmorphism .skill-progressbar-e-square-fill {
            z-index: 4;
        }
        
        .skill-glassmorphism .skill-progressbar-e-square-glow {
            z-index: 5;
        }
        
        /* Gradient Style - 2025 Modern */
        .skill-gradient .skill-progressbar-e-item {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 24px;
            padding: 28px;
            box-shadow: 0 12px 40px rgba(102, 126, 234, 0.3);
            border: none;
            position: relative;
            overflow: hidden;
            transition: all 0.4s ease;
            display: block;
        }
        
        .skill-gradient .skill-progressbar-e-item::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            opacity: 0;
            transition: opacity 0.4s ease;
            z-index: 1;
            pointer-events: none;
        }
        
        .skill-gradient .skill-progressbar-e-item:hover::after {
            opacity: 1;
        }
        
        .skill-gradient .skill-progressbar-e-content {
            position: relative;
            z-index: 2;
        }
        
        .skill-gradient .skill-progressbar-e-title,
        .skill-gradient .skill-progressbar-e-percent {
            color: white !important;
        }
        
        /* Ensure squares stay visible in gradient style */
        .skill-gradient .skill-progressbar-e-square {
            background: rgba(255, 255, 255, 0.2) !important;
            border: 1px solid rgba(255, 255, 255, 0.3) !important;
            position: relative;
            z-index: 3;
        }
        
        .skill-gradient .skill-progressbar-e-square-fill {
            z-index: 4;
        }
        
        .skill-gradient .skill-progressbar-e-square-glow {
            z-index: 5;
        }
        
        /* Neumorphism Style - 2025 Modern */
        .skill-neumorphism .skill-progressbar-e-item {
            background: #f0f0f3;
            border-radius: 24px;
            padding: 28px;
            box-shadow: 
                12px 12px 24px #d1d1d4,
                -12px -12px 24px #ffffff;
            border: none;
            transition: all 0.3s ease;
            display: block;
        }
        
        .skill-neumorphism .skill-progressbar-e-item:hover {
            box-shadow: 
                inset 12px 12px 24px #d1d1d4,
                inset -12px -12px 24px #ffffff;
            transform: translateY(2px);
        }
        
        /* Ensure squares stay visible in neumorphism style */
        .skill-neumorphism .skill-progressbar-e-square {
            background: #e0e5ec !important;
            box-shadow: 
                inset 4px 4px 8px #a3b1c6,
                inset -4px -4px 8px #ffffff !important;
            border: none !important;
            position: relative;
            z-index: 3;
        }
        
        .skill-neumorphism .skill-progressbar-e-square-fill {
            z-index: 4;
        }
        
        .skill-neumorphism .skill-progressbar-e-square-glow {
            z-index: 5;
        }
        
        /* Minimal Style - 2025 Modern */
        .skill-minimal .skill-progressbar-e-item {
            background: #ffffff;
            border-radius: 20px;
            padding: 24px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            border: 1px solid #f0f0f0;
            transition: all 0.3s ease;
            display: block;
        }
        
        .skill-minimal .skill-progressbar-e-item:hover {
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
            transform: translateY(-4px);
        }
        
        /* Ensure squares stay visible in minimal style */
        .skill-minimal .skill-progressbar-e-square {
            background: #f5f5f5 !important;
            border: 1px solid #e0e0e0 !important;
            position: relative;
            z-index: 3;
        }
        
        .skill-minimal .skill-progressbar-e-square-fill {
            z-index: 4;
        }
        
        .skill-minimal .skill-progressbar-e-square-glow {
            z-index: 5;
        }
        
        /* Common Styles */
        .skill-progressbar-e-title {
            font-size: 18px;
            font-weight: 600;
            margin-bottom: 8px;
            color: #333;
        }
        
        .skill-progressbar-e-percent {
            font-size: 16px;
            font-weight: 500;
            color: #666;
            margin-bottom: 16px;
        }
        
        .skill-progressbar-e-squares {
            display: flex;
            gap: 8px;
            align-items: center;
        }
        
        .skill-progressbar-e-square {
            width: 16px;
            height: 16px;
            border-radius: 4px;
            position: relative;
            overflow: hidden;
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            display: block;
            background: #f5f5f5 !important;
            border: 1px solid #e0e0e0;
        }
        
        .skill-progressbar-e-square-fill {
            position: absolute;
            top: 0;
            left: 0;
            height: 100%;
            background: linear-gradient(135deg, #667eea, #764ba2);
            transition: width 0.6s ease;
            z-index: 1;
        }
        
        .skill-progressbar-e-square-glow {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: linear-gradient(45deg, transparent, rgba(255, 255, 255, 0.3), transparent);
            animation: glowMove 2s ease-in-out infinite;
            z-index: 2;
        }
        
        /* Force background visibility for all widget instances */
        .skill-progressbar-e-widget .skill-progressbar-e-square {
            background: #f5f5f5 !important;
            border: 1px solid #e0e0e0 !important;
            display: block !important;
            position: relative !important;
            z-index: 3 !important;
        }
        
        /* Specific styles for each design */
        .skill-progressbar-e-widget.skill-glassmorphism .skill-progressbar-e-square {
            background: rgba(255, 255, 255, 0.2) !important;
            border: 1px solid rgba(255, 255, 255, 0.3) !important;
        }
        
        .skill-progressbar-e-widget.skill-neumorphism .skill-progressbar-e-square {
            background: #e0e5ec !important;
            box-shadow: inset 4px 4px 8px #a3b1c6, inset -4px -4px 8px #ffffff !important;
            border: none !important;
        }
        
        .skill-progressbar-e-widget.skill-gradient .skill-progressbar-e-square {
            background: rgba(255, 255, 255, 0.2) !important;
            border: 1px solid rgba(255, 255, 255, 0.3) !important;
        }
        
        .skill-progressbar-e-widget.skill-minimal .skill-progressbar-e-square {
            background: #f5f5f5 !important;
            border: 1px solid #e0e0e0 !important;
        }
        
        /* Ensure all squares are visible in Elementor editor */
        .elementor-editor-active .skill-progressbar-e-widget .skill-progressbar-e-square {
            opacity: 1 !important;
            visibility: visible !important;
            display: block !important;
            background: #f5f5f5 !important;
            border: 1px solid #e0e0e0 !important;
        }
        
        .elementor-editor-active .skill-progressbar-e-widget.skill-glassmorphism .skill-progressbar-e-square {
            background: rgba(255, 255, 255, 0.2) !important;
            border: 1px solid rgba(255, 255, 255, 0.3) !important;
        }
        
        .elementor-editor-active .skill-progressbar-e-widget.skill-neumorphism .skill-progressbar-e-square {
            background: #e0e5ec !important;
            box-shadow: inset 4px 4px 8px #a3b1c6, inset -4px -4px 8px #ffffff !important;
            border: none !important;
        }
        
        .elementor-editor-active .skill-progressbar-e-widget.skill-gradient .skill-progressbar-e-square {
            background: rgba(255, 255, 255, 0.2) !important;
            border: 1px solid rgba(255, 255, 255, 0.3) !important;
        }
        
        .elementor-editor-active .skill-progressbar-e-widget.skill-minimal .skill-progressbar-e-square {
            background: #f5f5f5 !important;
            border: 1px solid #e0e0e0 !important;
        }
        
        /* Animation Classes - Only apply when animations are enabled */
        .skill-progressbar-e-widget[data-animations="enabled"] .skill-progressbar-e-item {
            opacity: 1;
            transform: translateY(0);
        }
        
        /* Keyframe Animations */
        @keyframes slideInUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        @keyframes fillSquare {
            from {
                width: 0%;
            }
            to {
                width: 100%;
            }
        }
        
        @keyframes glowMove {
            0%, 100% {
                transform: translateX(-100%);
            }
            50% {
                transform: translateX(100%);
            }
        }
        
        /* Responsive Design */
        @media (max-width: 768px) {
            .skill-progressbar-e-items {
                grid-template-columns: 1fr;
                gap: 20px;
            }
            
            .skill-progressbar-e-item {
                padding: 24px;
            }
            
            .skill-progressbar-e-squares {
                gap: 10px;
            }
            
            .skill-progressbar-e-square {
                width: 28px;
                height: 28px;
            }
            
            .skill-progressbar-e-title {
                font-size: 16px;
            }
            
            .skill-progressbar-e-percent {
                font-size: 18px;
            }
        }
        
        /* Modern hover effects */
        .skill-progressbar-e-item {
            cursor: pointer;
        }
        
        .skill-progressbar-e-item:hover .skill-progressbar-e-title {
            transform: translateX(4px);
        }
        
        .skill-progressbar-e-item:hover .skill-progressbar-e-percent {
            transform: scale(1.1);
        }
        
        /* Elementor Editor Specific Styles */
        .elementor-editor-active .skill-progressbar-e-widget[data-animations="enabled"] .skill-progressbar-e-item {
            opacity: 1 !important;
            transform: translateY(0) !important;
            animation: none !important;
        }
        
        /* Force visibility in Elementor */
        .elementor-editor-active .skill-progressbar-e-widget {
            display: block !important;
        }
        
        .elementor-editor-active .skill-progressbar-e-items {
            display: grid !important;
        }
        
        .elementor-editor-active .skill-progressbar-e-item {
            display: block !important;
            opacity: 1 !important;
            visibility: visible !important;
        }
        </style>
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            const widgets = document.querySelectorAll('.skill-progressbar-e-widget');
            
            widgets.forEach(function(widget) {
                const animationsEnabled = widget.getAttribute('data-animations') === 'enabled';
                const isElementorEditor = document.body.classList.contains('elementor-editor-active');
                const designStyle = widget.className.match(/skill-(\w+)/);
                const currentStyle = designStyle ? designStyle[1] : 'minimal';
                
                function ensureItemsVisible() {
                    const items = widget.querySelectorAll('.skill-progressbar-e-item');
                    const squares = widget.querySelectorAll('.skill-progressbar-e-square');
                    
                    items.forEach(item => {
                        item.style.opacity = '1';
                        item.style.visibility = 'visible';
                        item.style.display = 'block';
                        item.style.transform = 'translateY(0)';
                    });
                    
                    squares.forEach(square => {
                        square.style.opacity = '1';
                        square.style.visibility = 'visible';
                        square.style.display = 'block';
                        square.style.position = 'relative';
                        square.style.zIndex = '3';
                        
                        // Force background color based on style
                        switch(currentStyle) {
                            case 'glassmorphism':
                                square.style.backgroundColor = 'rgba(255, 255, 255, 0.2)';
                                square.style.border = '1px solid rgba(255, 255, 255, 0.3)';
                                break;
                            case 'neumorphism':
                                square.style.backgroundColor = '#e0e5ec';
                                square.style.border = 'none';
                                square.style.boxShadow = 'inset 4px 4px 8px #a3b1c6, inset -4px -4px 8px #ffffff';
                                break;
                            case 'gradient':
                                square.style.backgroundColor = 'rgba(255, 255, 255, 0.2)';
                                square.style.border = '1px solid rgba(255, 255, 255, 0.3)';
                                break;
                            case 'minimal':
                                square.style.backgroundColor = '#f5f5f5';
                                square.style.border = '1px solid #e0e0e0';
                                break;
                            default:
                                square.style.backgroundColor = '#f5f5f5';
                                square.style.border = '1px solid #e0e0e0';
                        }
                    });
                }
                
                // Ensure visibility immediately
                ensureItemsVisible();
                
                if (animationsEnabled && !isElementorEditor) {
                    const observer = new IntersectionObserver((entries) => {
                        entries.forEach(entry => {
                            if (entry.isIntersecting) {
                                const item = entry.target;
                                item.style.animation = 'slideInUp 0.8s ease forwards';
                                
                                // Animate squares with delay
                                const squares = item.querySelectorAll('.skill-progressbar-e-square');
                                squares.forEach((square, index) => {
                                    setTimeout(() => {
                                        square.style.animation = 'fillSquare 0.6s ease forwards';
                                    }, 200 + (index * 100));
                                });
                            }
                        });
                    }, { threshold: 0.3 });
                    
                    const items = widget.querySelectorAll('.skill-progressbar-e-item');
                    items.forEach(item => observer.observe(item));
                    
                    // Hover effects for squares
                    const squares = widget.querySelectorAll('.skill-progressbar-e-square');
                    squares.forEach(square => {
                        square.addEventListener('mouseenter', function() {
                            this.style.transform = 'scale(1.1)';
                            this.style.boxShadow = '0 8px 25px rgba(0, 0, 0, 0.15)';
                        });
                        
                        square.addEventListener('mouseleave', function() {
                            this.style.transform = 'scale(1)';
                            this.style.boxShadow = '';
                        });
                    });
                }
                
                // Additional checks for frontend
                if (!isElementorEditor) {
                    setTimeout(ensureItemsVisible, 100);
                    setTimeout(ensureItemsVisible, 500);
                    setTimeout(ensureItemsVisible, 1000);
                }
            });
        });
        </script>
        <?php
    }
    
    private function adjust_brightness($hex, $steps) {
        $hex = str_replace('#', '', $hex);
        
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        
        $r = max(0, min(255, $r + $steps));
        $g = max(0, min(255, $g + $steps));
        $b = max(0, min(255, $b + $steps));
        
        return sprintf("#%02x%02x%02x", $r, $g, $b);
    }
}