<?php
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class skill_collaboration_Widget extends Widget_Base {
    public function get_name() {
        return 'skill_collaboration';
    }

    public function get_title() {
        return __( 'همکاری‌های من', 'skill' );
    }

    public function get_icon() {
        return 'eicon-gallery-grid';
    }

    public function get_categories() {
        return [ 'skill-category' ];
    }

    protected function _register_controls() {
        // --- Content Tab ---
        $this->start_controls_section(
            'section_title',
            [
                'label' => __( 'عنوان', 'skill' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'show_title',
            [
                'label' => __( 'نمایش عنوان', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __( 'بله', 'skill' ),
                'label_off' => __( 'خیر', 'skill' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'title_text',
            [
                'label' => __( 'عنوان', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'همکاری‌های من', 'skill' ),
                'placeholder' => __( 'عنوان را وارد کنید', 'skill' ),
                'label_block' => true,
                'dynamic' => [ 'active' => true ],
                'condition' => [ 'show_title' => 'yes' ],
            ]
        );
        $this->add_control(
            'show_dot',
            [
                'label' => __( 'نمایش مربع آبی', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __( 'بله', 'skill' ),
                'label_off' => __( 'خیر', 'skill' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'dot_position',
            [
                'label' => __( 'جایگاه مربع', 'skill' ),
                'type' => Controls_Manager::SELECT,
                'options' => [ 'right' => __( 'راست عنوان', 'skill' ), 'left' => __( 'چپ عنوان', 'skill' ) ],
                'default' => 'right',
                'condition' => [ 'show_dot' => 'yes' ],
            ]
        );
        $this->add_control(
            'show_desc',
            [
                'label' => __( 'نمایش توضیح کوتاه', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __( 'بله', 'skill' ),
                'label_off' => __( 'خیر', 'skill' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'desc_text',
            [
                'label' => __( 'توضیح کوتاه', 'skill' ),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 2,
                'default' => '',
                'condition' => [ 'show_desc' => 'yes' ],
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'section_collaborations',
            [
                'label' => __( 'همکاری‌ها', 'skill' ),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $repeater = new Repeater();
        $repeater->add_control(
            'company_logo',
            [
                'label' => __( 'لوگو شرکت', 'skill' ),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );
        $repeater->add_control(
            'company_name',
            [
                'label' => __( 'نام شرکت', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'نام شرکت', 'skill' ),
                'label_block' => true,
            ]
        );
        $repeater->add_control(
            'company_url',
            [
                'label' => __( 'لینک وب‌سایت', 'skill' ),
                'type' => Controls_Manager::URL,
                'placeholder' => 'https://example.com',
                'default' => [ 'url' => '', 'is_external' => true, 'nofollow' => false ],
            ]
        );
        $repeater->add_control(
            'logo_alt',
            [
                'label' => __( 'متن جایگزین لوگو (alt)', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => '',
            ]
        );
        $repeater->add_control(
            'role_title',
            [
                'label' => __( 'نقش/سمت شما', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'label_block' => true,
            ]
        );
        $repeater->add_control(
            'period_start',
            [
                'label' => __( 'شروع همکاری (مثلاً 1399)', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => '',
            ]
        );
        $repeater->add_control(
            'period_end',
            [
                'label' => __( 'پایان همکاری (مثلاً 1401)', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => '',
            ]
        );
        $repeater->add_control(
            'is_current',
            [
                'label' => __( 'هم‌اکنون مشغولم', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __( 'بله', 'skill' ),
                'label_off' => __( 'خیر', 'skill' ),
                'return_value' => 'yes',
                'default' => '',
            ]
        );
        $repeater->add_control(
            'description',
            [
                'label' => __( 'توضیح کوتاه', 'skill' ),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 3,
                'default' => '',
            ]
        );
        $repeater->add_control(
            'cta_text',
            [
                'label' => __( 'متن دکمه (اختیاری)', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => '',
            ]
        );
        $repeater->add_control(
            'work_mode',
            [
                'label' => __( 'نوع همکاری / حضور', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'label_block' => true,
            ]
        );
        $repeater->add_control(
            'work_hours',
            [
                'label' => __( 'ساعت کاری', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'label_block' => true,
            ]
        );
        $repeater->add_control(
            'show_ended_status',
            [
                'label' => __( 'نمایش وضعیت اتمام همکاری', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __( 'بله', 'skill' ),
                'label_off' => __( 'خیر', 'skill' ),
                'return_value' => 'yes',
                'default' => '',
            ]
        );
        $repeater->add_control(
            'ended_text',
            [
                'label' => __( 'متن اتمام همکاری', 'skill' ),
                'type' => Controls_Manager::TEXT,
                'default' => __( 'اتمام همکاری', 'skill' ),
                'condition' => [ 'show_ended_status' => 'yes' ],
            ]
        );
        $this->add_control(
            'collaborations',
            [
                'label' => __( 'همکاری‌ها', 'skill' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'company_name' => __( 'Google', 'skill' ),
                        'company_logo' => [ 'url' => Elementor\Utils::get_placeholder_image_src() ],
                    ],
                    [
                        'company_name' => __( 'Microsoft', 'skill' ),
                        'company_logo' => [ 'url' => Elementor\Utils::get_placeholder_image_src() ],
                    ],
                ],
                'title_field' => '{{{ company_name }}}',
            ]
        );
        $this->add_control(
            'show_description',
            [
                'label' => __( 'نمایش توضیح', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'label_on' => __( 'بله', 'skill' ),
                'label_off' => __( 'خیر', 'skill' ),
            ]
        );
        $this->add_control(
            'show_period',
            [
                'label' => __( 'نمایش بازه زمانی', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'show_cta',
            [
                'label' => __( 'نمایش دکمه مشاهده سایت', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no',
            ]
        );
        $this->add_control(
            'link_whole_card',
            [
                'label' => __( 'کلیک‌پذیر بودن کل کارت', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no',
            ]
        );
        $this->add_control(
            'enable_popup',
            [
                'label' => __( 'باز شدن پاپ‌آپ با کلیک روی کارت', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'description' => __( 'در صورت فعال بودن، به جای لینک، اطلاعات کامل در پاپ‌آپ نمایش داده می‌شود.', 'skill' ),
            ]
        );

        // Background Vectors Controls
        $this->add_control(
            'show_bg_vectors',
            [
                'label' => __( 'نمایش وکتورهای پس‌زمینه', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'condition' => [
                    'enable_popup' => 'yes',
                ],
            ]
        );
        $this->add_control(
            'vector_1_color',
            [
                'label' => __( 'رنگ وکتور اول', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#4f8cff',
                'condition' => [
                    'enable_popup' => 'yes',
                    'show_bg_vectors' => 'yes',
                ],
                'selectors' => [
                    '{{WRAPPER}} .collab-modal__vector-1' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'vector_2_color',
            [
                'label' => __( 'رنگ وکتور دوم', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#6ee7b7',
                'condition' => [
                    'enable_popup' => 'yes',
                    'show_bg_vectors' => 'yes',
                ],
                'selectors' => [
                    '{{WRAPPER}} .collab-modal__vector-2' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'vector_3_color',
            [
                'label' => __( 'رنگ وکتور سوم', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ffb43a',
                'condition' => [
                    'enable_popup' => 'yes',
                    'show_bg_vectors' => 'yes',
                ],
                'selectors' => [
                    '{{WRAPPER}} .collab-modal__vector-3' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'vector_4_color',
            [
                'label' => __( 'رنگ وکتور چهارم', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ff6b6b',
                'condition' => [
                    'enable_popup' => 'yes',
                    'show_bg_vectors' => 'yes',
                ],
                'selectors' => [
                    '{{WRAPPER}} .collab-modal__vector-4' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'vector_opacity',
            [
                'label' => __( 'شفافیت وکتورها', 'skill' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ '%' ],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'size' => 15,
                    'unit' => '%',
                ],
                'condition' => [
                    'enable_popup' => 'yes',
                    'show_bg_vectors' => 'yes',
                ],
                'selectors' => [
                    '{{WRAPPER}} .collab-modal__vector' => 'opacity: {{SIZE}};',
                ],
            ]
        );

        // Animated Vectors Controls
        $this->add_control(
            'show_animated_vectors',
            [
                'label' => __( 'نمایش وکتورهای متحرک', 'skill' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'condition' => [
                    'enable_popup' => 'yes',
                    'show_bg_vectors' => 'yes',
                ],
            ]
        );
        $this->add_control(
            'animated_vectors_count',
            [
                'label' => __( 'تعداد وکتورهای متحرک', 'skill' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ '' ],
                'range' => [
                    '' => [
                        'min' => 3,
                        'max' => 12,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'size' => 8,
                    'unit' => '',
                ],
                'condition' => [
                    'enable_popup' => 'yes',
                    'show_bg_vectors' => 'yes',
                    'show_animated_vectors' => 'yes',
                ],
            ]
        );
        $this->add_control(
            'animation_speed',
            [
                'label' => __( 'سرعت انیمیشن', 'skill' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 's' ],
                'range' => [
                    's' => [
                        'min' => 10,
                        'max' => 60,
                        'step' => 1,
                    ],
                ],
                'default' => [
                    'size' => 25,
                    'unit' => 's',
                ],
                'condition' => [
                    'enable_popup' => 'yes',
                    'show_bg_vectors' => 'yes',
                    'show_animated_vectors' => 'yes',
                ],
                'selectors' => [
                    '{{WRAPPER}} .collab-modal__animated-vector' => 'animation-duration: {{SIZE}}s;',
                ],
            ]
        );
        $this->add_control(
            'order_mode',
            [
                'label' => __( 'مرتب‌سازی', 'skill' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'input',
                'options' => [
                    'input' => __( 'همان ترتیب ورودی', 'skill' ),
                    'az' => __( 'حروف الفبا (A→Z)', 'skill' ),
                    'random' => __( 'تصادفی', 'skill' ),
                ],
            ]
        );
        $this->add_responsive_control(
            'columns',
            [
                'label' => __( 'تعداد ستون‌ها', 'skill' ),
                'type' => Controls_Manager::SELECT,
                'options' => [ '1'=>'1', '2'=>'2', '3'=>'3', '4'=>'4', '5'=>'5', '6'=>'6' ],
                'default' => '4',
                'selectors' => [
                    '{{WRAPPER}} .collaboration-grid' => 'grid-template-columns: repeat({{VALUE}}, minmax(160px, 1fr));',
                ],
            ]
        );
        $this->end_controls_section();

        // --- Style Tab ---
        $this->start_controls_section(
            'section_title_style',
            [
                'label' => __( 'استایل عنوان', 'skill' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_responsive_control(
            'tdw_align',
            [
                'label' => __( 'چینش', 'skill' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'right' => [ 'title' => __( 'راست', 'skill' ), 'icon' => 'eicon-text-align-right' ],
                    'center'=> [ 'title' => __( 'وسط', 'skill' ), 'icon' => 'eicon-text-align-center' ],
                    'left'  => [ 'title' => __( 'چپ', 'skill' ), 'icon' => 'eicon-text-align-left' ],
                ],
                'default' => 'center',
                'selectors' => [ '{{WRAPPER}} .tdw-wrap' => 'text-align: {{VALUE}};' ],
            ]
        );
        $this->add_control(
            'title_color',
            [
                'label' => __( 'رنگ عنوان', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#1a1a1a',
                'selectors' => [
                    '{{WRAPPER}} .tdw-title' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .tdw-title',
            ]
        );
        $this->add_control(
            'desc_color',
            [
                'label' => __( 'رنگ توضیح', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => 'rgba(0,0,0,0.6)',
                'selectors' => [ '{{WRAPPER}} .tdw-desc' => 'color: {{VALUE}};' ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'desc_typography',
                'selector' => '{{WRAPPER}} .tdw-desc',
            ]
        );
        $this->add_control(
            'dot_color',
            [
                'label' => __( 'رنگ مربع', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#1e73be',
                'selectors' => [ '{{WRAPPER}} .tdw-dot' => 'background-color: {{VALUE}};' ],
            ]
        );
        $this->add_control(
            'dot_size',
            [
                'label' => __( 'سایز مربع (px)', 'skill' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [ 'px' => [ 'min' => 6, 'max' => 24 ] ],
                'default' => [ 'unit' => 'px', 'size' => 10 ],
                'selectors' => [ '{{WRAPPER}} .tdw-dot' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};' ],
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'section_item_style',
            [
                'label' => __( 'استایل آیتم همکاری', 'skill' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'item_bg_color',
            [
                'label' => __( 'رنگ پس‌زمینه آیتم', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [
                    '{{WRAPPER}} .collaboration-item' => 'background: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'item_border_radius',
            [
                'label' => __( 'گردی گوشه آیتم', 'skill' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [ 'min' => 0, 'max' => 50 ],
                ],
                'default' => [ 'size' => 18, 'unit' => 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .collaboration-item' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_box_shadow',
                'selector' => '{{WRAPPER}} .collaboration-item',
            ]
        );
        $this->add_responsive_control(
            'item_padding',
            [
                'label' => __( 'فاصله داخلی آیتم', 'skill' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'default' => [ 'top' => 24, 'right' => 12, 'bottom' => 16, 'left' => 12, 'unit' => 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .collaboration-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'section_logo_style',
            [
                'label' => __( 'استایل لوگو', 'skill' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'logo_bg_color',
            [
                'label' => __( 'رنگ پس‌زمینه لوگو', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#f5f7fa',
                'selectors' => [
                    '{{WRAPPER}} .collaboration-logo-wrap' => 'background: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'logo_size',
            [
                'label' => __( 'اندازه لوگو', 'skill' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [ 'px' => [ 'min' => 24, 'max' => 120 ] ],
                'default' => [ 'size' => 56, 'unit' => 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .collaboration-logo' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'logo_wrap_size',
            [
                'label' => __( 'اندازه قاب لوگو', 'skill' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [ 'px' => [ 'min' => 32, 'max' => 160 ] ],
                'default' => [ 'size' => 72, 'unit' => 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .collaboration-logo-wrap' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'section_name_style',
            [
                'label' => __( 'استایل نام شرکت', 'skill' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'name_color',
            [
                'label' => __( 'رنگ نام شرکت', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#222',
                'selectors' => [
                    '{{WRAPPER}} .collaboration-name' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'role_color',
            [
                'label' => __( 'رنگ نقش/سمت', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .collaboration-role' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .collab-modal__role' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'period_color',
            [
                'label' => __( 'رنگ بازه زمانی', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#666',
                'selectors' => [
                    '{{WRAPPER}} .collaboration-period' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'description_color',
            [
                'label' => __( 'رنگ توضیحات کوتاه', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#555',
                'selectors' => [
                    '{{WRAPPER}} .collaboration-desc' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'label' => __( 'تایپوگرافی توضیحات کوتاه', 'skill' ),
                'selector' => '{{WRAPPER}} .collaboration-desc',
            ]
        );
        $this->end_controls_section();

        // CTA style
        $this->start_controls_section(
            'section_cta_style',
            [
                'label' => __( 'استایل دکمه', 'skill' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'cta_bg',
            [
                'label' => __( 'رنگ پس‌زمینه دکمه', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ffb43a',
                'selectors' => [ '{{WRAPPER}} .collaboration-cta' => 'background: {{VALUE}};' ],
            ]
        );
        $this->add_control(
            'cta_text_color',
            [
                'label' => __( 'رنگ متن دکمه', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#232432',
                'selectors' => [ '{{WRAPPER}} .collaboration-cta' => 'color: {{VALUE}};' ],
            ]
        );
        $this->add_control(
            'cta_bg_hover',
            [
                'label' => __( 'رنگ پس‌زمینه (هاور)', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ff9800',
                'selectors' => [ '{{WRAPPER}} .collaboration-cta:hover' => 'background: {{VALUE}};' ],
            ]
        );
        $this->add_control(
            'cta_text_color_hover',
            [
                'label' => __( 'رنگ متن (هاور)', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#fff',
                'selectors' => [ '{{WRAPPER}} .collaboration-cta:hover' => 'color: {{VALUE}};' ],
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'section_ended_status_style',
            [
                'label' => __( 'استایل وضعیت اتمام', 'skill' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'ended_bg_color',
            [
                'label' => __( 'رنگ پس‌زمینه', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ff6b6b',
                'selectors' => [
                    '{{WRAPPER}} .ended-badge' => 'background: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'ended_text_color',
            [
                'label' => __( 'رنگ متن', 'skill' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .ended-badge' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'ended_border_radius',
            [
                'label' => __( 'گردی گوشه', 'skill' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [ 'min' => 0, 'max' => 50 ],
                ],
                'default' => [ 'size' => 12, 'unit' => 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .ended-badge' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'ended_font_size',
            [
                'label' => __( 'اندازه فونت', 'skill' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [ 'min' => 8, 'max' => 16 ],
                ],
                'default' => [ 'size' => 11, 'unit' => 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .ended-badge' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'section_grid_style',
            [
                'label' => __( 'استایل گرید', 'skill' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_responsive_control(
            'grid_gap',
            [
                'label' => __( 'فاصله بین آیتم‌ها', 'skill' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [ 'px' => [ 'min' => 0, 'max' => 64 ] ],
                'default' => [ 'size' => 24, 'unit' => 'px' ],
                'selectors' => [
                    '{{WRAPPER}} .collaboration-grid' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        if ( empty( $settings['collaborations'] ) ) return;

        // Title style determination logic
        $title_style = '';
        $has_custom_color = !empty($settings['title_color']);
        if (!$has_custom_color) {
            $title_style = 'background: linear-gradient(90deg, #4f8cff 0%, #6ee7b7 100%); -webkit-background-clip: text; -webkit-text-fill-color: transparent; background-clip: text;';
        }

        // Ordering
        $items = $settings['collaborations'];
        switch ( $settings['order_mode'] ?? 'input' ) {
            case 'az':
                usort($items, function($a, $b){
                    return strcasecmp($a['company_name'] ?? '', $b['company_name'] ?? '');
                });
                break;
            case 'random':
                shuffle($items);
                break;
        }

        $use_popup = !empty($settings['enable_popup']);
        $widget_id = $this->get_id();

        ?>
        <div class="skill-collaboration-widget" id="skill-collab-<?php echo esc_attr($widget_id); ?>">
            <div class="tdw-wrap" dir="rtl">
                <div class="tdw-title-row" style="display:inline-flex; align-items:center; gap:8px; <?php echo (($settings['dot_position'] ?? 'right') === 'right') ? 'flex-direction: row-reverse;' : ''; ?>">
                    <?php if (!empty($settings['show_title']) && $settings['show_title'] === 'yes' && !empty($settings['title_text'])) : ?>
                        <h2 class="tdw-title"><?php echo esc_html($settings['title_text']); ?></h2>
                    <?php endif; ?>
                    <?php if (!empty($settings['show_dot']) && $settings['show_dot'] === 'yes') : ?>
                        <span class="tdw-dot" aria-hidden="true" style="display:inline-block; border-radius:3px;"></span>
                    <?php endif; ?>
                </div>
                <?php if (!empty($settings['show_desc']) && $settings['show_desc'] === 'yes' && !empty($settings['desc_text'])) : ?>
                    <div class="tdw-desc" style="margin-top:10px;">
                        <?php echo esc_html($settings['desc_text']); ?>
                    </div>
                <?php endif; ?>
            </div>
            <div class="collaboration-grid">
                <?php foreach ( $items as $index => $item ) :
                    $company_name = esc_html( $item['company_name'] ?? '' );
                    $logo_url = isset($item['company_logo']['url']) ? $item['company_logo']['url'] : '';
                    $logo_alt = !empty($item['logo_alt']) ? $item['logo_alt'] : $company_name;
                    $link = $item['company_url']['url'] ?? '';
                    $is_external = !empty($item['company_url']['is_external']);
                    $nofollow = !empty($item['company_url']['nofollow']);
                    $role = esc_html( $item['role_title'] ?? '' );
                    $start = trim($item['period_start'] ?? '');
                    $end = trim($item['period_end'] ?? '');
                    $is_current = !empty($item['is_current']);
                    $desc = isset($item['description']) ? wp_kses_post($item['description']) : '';
                    $cta_text = trim($item['cta_text'] ?? '');
                    $work_mode = esc_html( $item['work_mode'] ?? '' );
                    $work_hours = esc_html( $item['work_hours'] ?? '' );
                    $show_ended_status = !empty($item['show_ended_status']);
                    $ended_text = esc_html( $item['ended_text'] ?? __( 'اتمام همکاری', 'skill' ) );

                    $period_text = '';
                    if ( !empty($settings['show_period']) ) {
                        if ($start || $end || $is_current) {
                            $period_text = esc_html($start);
                            if ($is_current) {
                                $period_text .= ($period_text ? ' — ' : '') . esc_html__('هم‌اکنون', 'skill');
                            } elseif ($end) {
                                $period_text .= ($period_text ? ' — ' : '') . esc_html($end);
                            }
                        }
                    }

                    $card_open = '';
                    $card_close = '';
                    $modal_id = 'collab-modal-' . esc_attr($widget_id) . '-' . $index;
                    if ( empty($use_popup) && !empty($settings['link_whole_card']) && !empty($link) ) {
                        $rel = $nofollow ? 'nofollow noopener' : 'noopener';
                        $target = $is_external ? ' target="_blank"' : '';
                        $card_open = '<a class="collaboration-item-link" href="' . esc_url($link) . '" rel="' . esc_attr($rel) . '"' . $target . '>';
                        $card_close = '</a>';
                    }
                ?>
                <article class="collaboration-item<?php echo $use_popup ? ' js-open-collab-modal' : ''; ?>" aria-label="<?php echo esc_attr($company_name); ?>" <?php echo $use_popup ? 'data-modal-id="'.$modal_id.'"' : ''; ?>>
                    <?php echo $card_open; ?>
                    <?php if ($show_ended_status): ?>
                        <div class="collaboration-ended-status">
                            <span class="ended-badge"><?php echo $ended_text; ?></span>
                        </div>
                    <?php endif; ?>
                    <div class="collaboration-content">
                        <div class="collaboration-logo-wrap">
                            <?php if ($logo_url): ?>
                                <img class="collaboration-logo" src="<?php echo esc_url( $logo_url ); ?>" alt="<?php echo esc_attr( $logo_alt ); ?>" loading="lazy" decoding="async" width="112" height="112" />
                            <?php endif; ?>
                        </div>
                        <div class="collaboration-info">
                            <h3 class="collaboration-name" title="<?php echo esc_attr($company_name); ?>"><?php echo $company_name; ?></h3>
                            <?php if ($role): ?><div class="collaboration-role"><i class="fa fa-user" aria-hidden="true"></i><?php echo esc_html($role); ?></div><?php endif; ?>
                            <?php if ($period_text): ?><div class="collaboration-period"><i class="fa fa-calendar-alt" aria-hidden="true"></i><?php echo $period_text; ?></div><?php endif; ?>
                        </div>
                    </div>
                    <?php if ( !empty($settings['show_description']) && $desc ): ?><div class="collaboration-desc"><?php echo $desc; ?></div><?php endif; ?>
                    <?php if ( empty($use_popup) && !empty($settings['show_cta']) && !empty($cta_text) && !empty($link) && empty($settings['link_whole_card']) ): ?>
                        <a class="collaboration-cta" href="<?php echo esc_url($link); ?>"<?php echo $is_external ? ' target="_blank"' : ''; ?> rel="<?php echo esc_attr( $nofollow ? 'nofollow noopener' : 'noopener' ); ?>"><?php echo esc_html($cta_text); ?></a>
                    <?php endif; ?>
                    <?php echo $card_close; ?>
                </article>

                <?php if ( $use_popup ) : ?>
                    <div id="<?php echo esc_attr($modal_id); ?>" class="collab-modal" aria-hidden="true">
                        <div class="collab-modal__overlay" data-close="1"></div>
                        <div class="collab-modal__dialog" role="dialog" aria-modal="true" aria-labelledby="<?php echo esc_attr($modal_id); ?>-title">
                            <?php if ( !empty($settings['show_bg_vectors']) ) : ?>
                                <!-- Background Vectors -->
                                <svg class="collab-modal__vector collab-modal__vector-1" viewBox="0 0 100 100" aria-hidden="true">
                                    <circle cx="20" cy="20" r="8" fill="currentColor"/>
                                </svg>
                                <svg class="collab-modal__vector collab-modal__vector-2" viewBox="0 0 100 100" aria-hidden="true">
                                    <path d="M80 15 L90 25 L80 35 L70 25 Z" fill="currentColor"/>
                                </svg>
                                <svg class="collab-modal__vector collab-modal__vector-3" viewBox="0 0 100 100" aria-hidden="true">
                                    <rect x="15" y="70" width="12" height="12" rx="2" fill="currentColor"/>
                                </svg>
                                <svg class="collab-modal__vector collab-modal__vector-4" viewBox="0 0 100 100" aria-hidden="true">
                                    <polygon points="85,75 90,85 80,85" fill="currentColor"/>
                                </svg>

                                <?php if ( !empty($settings['show_animated_vectors']) ) : ?>
                                    <!-- Animated Background Vectors -->
                                    <?php 
                                    $animated_count = !empty($settings['animated_vectors_count']['size']) ? intval($settings['animated_vectors_count']['size']) : 8;
                                    for ($i = 1; $i <= $animated_count; $i++) : 
                                        $vector_type = ($i % 4) + 1;
                                        $delay = ($i * 0.5) . 's';
                                        $size = 20 + ($i % 3) * 8;
                                        $top = rand(5, 85);
                                        $left = rand(5, 85);
                                    ?>
                                        <svg class="collab-modal__animated-vector collab-modal__animated-vector-<?php echo $vector_type; ?>" 
                                             viewBox="0 0 100 100" 
                                             aria-hidden="true"
                                             style="top: <?php echo $top; ?>%; left: <?php echo $left; ?>%; width: <?php echo $size; ?>px; height: <?php echo $size; ?>px; animation-delay: <?php echo $delay; ?>;">
                                            <?php if ($vector_type == 1): ?>
                                                <circle cx="50" cy="50" r="25" fill="currentColor"/>
                                            <?php elseif ($vector_type == 2): ?>
                                                <path d="M25 25 L75 25 L75 75 L25 75 Z" fill="currentColor"/>
                                            <?php elseif ($vector_type == 3): ?>
                                                <polygon points="50,20 80,80 20,80" fill="currentColor"/>
                                            <?php else: ?>
                                                <ellipse cx="50" cy="50" rx="30" ry="20" fill="currentColor"/>
                                            <?php endif; ?>
                                        </svg>
                                    <?php endfor; ?>
                                <?php endif; ?>
                            <?php endif; ?>
                            <button class="collab-modal__close" aria-label="بستن" data-close="1">&times;</button>
                            <div class="collab-modal__header">
                                <div class="collab-modal__logo-wrap">
                                    <?php if ($logo_url): ?>
                                        <img class="collab-modal__logo" src="<?php echo esc_url( $logo_url ); ?>" alt="<?php echo esc_attr( $logo_alt ); ?>" width="112" height="112" />
                                    <?php endif; ?>
                                </div>
                                <div class="collab-modal__titles">
                                    <h2 id="<?php echo esc_attr($modal_id); ?>-title" class="collab-modal__company"><?php echo $company_name; ?></h2>
                                    <?php if ($role): ?><div class="collab-modal__role"><?php echo esc_html($role); ?></div><?php endif; ?>
                                </div>
                            </div>
                            <div class="collab-modal__body">
                                <div class="collab-modal__desc"><?php echo $desc; ?></div>
                                <ul class="collab-modal__facts">
                                    <?php if ($period_text): ?><li><i class="fa fa-calendar-alt" aria-hidden="true"></i><span><?php echo $period_text; ?></span></li><?php endif; ?>
                                    <?php if ($role): ?><li><i class="fa fa-user-tie" aria-hidden="true"></i><span><?php echo esc_html($role); ?></span></li><?php endif; ?>
                                    <?php if ($work_mode): ?><li><i class="fa fa-map-marker-alt" aria-hidden="true"></i><span><?php echo esc_html($work_mode); ?></span></li><?php endif; ?>
                                    <?php if ($work_hours): ?><li><i class="fa fa-clock" aria-hidden="true"></i><span><?php echo esc_html($work_hours); ?></span></li><?php endif; ?>
                                </ul>
                                <?php if ( !empty($link) ): ?>
                                    <div class="collab-modal__actions">
                                        <a class="collaboration-cta" href="<?php echo esc_url($link); ?>"<?php echo $is_external ? ' target="_blank"' : ''; ?> rel="<?php echo esc_attr( $nofollow ? 'nofollow noopener' : 'noopener' ); ?>"><?php echo !empty($cta_text) ? esc_html($cta_text) : esc_html__('مشاهده وب‌سایت', 'skill'); ?></a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>

                <?php endforeach; ?>
            </div>
        </div>
        <style>
        .skill-collaboration-widget {
            padding: 20px 0;
        }
        .collaboration-widget-title {
            text-align: center;
            font-size: 2em;
            font-weight: 800;
            margin-bottom: 32px;
            letter-spacing: 0.01em;
        }
        .collaboration-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(160px, 1fr));
            gap: 32px 24px;
            align-items: start;
            margin-top: 20px;
        }
        .collaboration-item {
            transition: transform 0.3s cubic-bezier(.4,2,.6,1), box-shadow 0.3s;
            background: #fff;
            border-radius: 18px;
            box-shadow: 0 2px 16px 0 rgba(0,0,0,0.07);
            padding: 20px 16px 16px 16px;
            position: relative;
            cursor: pointer;
        }
        
        /* Ended Status Badge */
        .collaboration-ended-status {
            position: absolute;
            top: 12px;
            left: 12px;
            z-index: 2;
        }
        .ended-badge {
            display: inline-block;
            background: linear-gradient(135deg, #ff6b6b 0%, #ee5a52 100%);
            color: #fff;
            font-size: 11px;
            font-weight: 700;
            padding: 6px 12px;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(255, 107, 107, 0.3);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        .collaboration-item:hover { transform: translateY(-8px) scale(1.02); box-shadow: 0 8px 32px 0 rgba(0,0,0,0.13); }
        .collaboration-item-link { position:absolute; inset:0; z-index:1; }
        
        /* Horizontal layout container */
        .collaboration-content {
            display: flex;
            align-items: center;
            gap: 16px;
            margin-bottom: 12px;
        }
        
        .collaboration-logo-wrap { 
            width: 72px; 
            height: 72px; 
            border-radius: 12px; 
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%); 
            display:flex; 
            align-items:center; 
            justify-content:center; 
            box-shadow: 0 2px 8px 0 rgba(0,0,0,0.08); 
            overflow:hidden; 
            transition: box-shadow 0.3s; 
            flex-shrink: 0;
        }
        .collaboration-item:hover .collaboration-logo-wrap { box-shadow: 0 4px 16px 0 rgba(0,0,0,0.16); }
        .collaboration-logo { width: 56px; height: 56px; object-fit: contain; border-radius: 8px; filter: grayscale(0.2) contrast(1.1); transition: filter 0.3s; }
        .collaboration-item:hover .collaboration-logo { filter: grayscale(0) contrast(1.2); }
        
        /* Text content container */
        .collaboration-info {
            flex: 1;
            text-align: right;
        }
        
        .collaboration-name { 
            margin: 0 0 8px 0; 
            font-size: 1.1em; 
            font-weight: 700; 
            letter-spacing: 0.01em; 
            color: #222;
            line-height: 1.3;
        }
        .collaboration-role { 
            margin: 0 0 6px 0; 
            font-size: .9em; 
            opacity: .8; 
            color: #666;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        .collaboration-role i { 
            font-size: 12px; 
            color: #4f8cff; 
        }
        .collaboration-period { 
            margin: 0; 
            font-size: .85em; 
            opacity: .7; 
            color: #888;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        .collaboration-period i { 
            font-size: 12px; 
            color: #4f8cff; 
        }
        .collaboration-desc { margin-top: 8px; font-size: .95em; opacity: .9; display:-webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical; overflow:hidden; text-overflow:ellipsis; min-height: 2.8em; }
        .collaboration-cta { margin-top: 12px; display:inline-block; padding: 8px 14px; border-radius: 10px; font-weight: 700; text-decoration: none; box-shadow: 0 2px 8px rgba(0,0,0,.08); position: relative; z-index:2; }
        @media (max-width: 600px) {
            .collaboration-grid { gap: 18px 12px; }
            .collaboration-content { gap: 12px; }
            .collaboration-logo-wrap { width: 60px; height: 60px; border-radius: 10px; }
            .collaboration-logo { width: 46px; height: 46px; border-radius: 6px; }
            .collaboration-name { font-size: 1em; }
            .collaboration-role { font-size: .85em; }
            .collaboration-period { font-size: .8em; }
            .ended-badge { font-size: 10px; padding: 4px 8px; }
        }

        /* Modal */
        .collab-modal { 
            position: fixed; 
            inset: 0; 
            z-index: 9999; 
            align-items:center; 
            justify-content:center; 
            opacity: 0;
            visibility: hidden;
            transition: opacity 0.3s ease, visibility 0.3s ease;
        }
        .collab-modal.is-open { 
            opacity: 1;
            visibility: visible;
        }
        .collab-modal__overlay { 
            position:absolute; 
            inset:0; 
            background: rgba(0,0,0,.6); 
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        .collab-modal.is-open .collab-modal__overlay { 
            opacity: 1;
        }
        .collab-modal__dialog { 
            position:relative; 
            margin: 0; 
            background: #121212; 
            color:#fff; 
            width:min(920px, 92vw); 
            border-radius: 20px; 
            box-shadow: 0 20px 80px rgba(0,0,0,.5); 
            overflow:hidden; 
            z-index:1; 
            transform: translateY(100vh);
            transition: transform 0.4s cubic-bezier(0.25, 0.46, 0.45, 0.94);
        }
        .collab-modal.is-open .collab-modal__dialog { 
            transform: translateY(0); 
        }
        
        /* Background Vectors */
        .collab-modal__vector { position: absolute; pointer-events: none; z-index: 0; opacity: 0.15; }
        .collab-modal__vector-1 { top: 10%; left: 10%; width: 60px; height: 60px; }
        .collab-modal__vector-2 { top: 15%; right: 15%; width: 50px; height: 50px; }
        .collab-modal__vector-3 { bottom: 20%; left: 8%; width: 45px; height: 45px; }
        .collab-modal__vector-4 { bottom: 15%; right: 10%; width: 55px; height: 55px; }

        /* Animated Background Vectors */
        .collab-modal__animated-vector { 
            position: absolute; 
            pointer-events: none; 
            z-index: 0; 
            opacity: 0.08; 
            animation: float-around 25s infinite linear;
        }
        
        @keyframes float-around {
            0% { 
                transform: translate(0, 0) rotate(0deg) scale(1); 
                opacity: 0.08;
            }
            25% { 
                transform: translate(20px, -15px) rotate(90deg) scale(1.1); 
                opacity: 0.12;
            }
            50% { 
                transform: translate(-10px, 25px) rotate(180deg) scale(0.9); 
                opacity: 0.06;
            }
            75% { 
                transform: translate(15px, 10px) rotate(270deg) scale(1.05); 
                opacity: 0.1;
            }
            100% { 
                transform: translate(0, 0) rotate(360deg) scale(1); 
                opacity: 0.08;
            }
        }

        /* Different animation patterns for different vector types */
        .collab-modal__animated-vector-1 { animation-timing-function: ease-in-out; }
        .collab-modal__animated-vector-2 { animation-timing-function: ease-in; }
        .collab-modal__animated-vector-3 { animation-timing-function: ease-out; }
        .collab-modal__animated-vector-4 { animation-timing-function: cubic-bezier(0.4, 0, 0.2, 1); }
        
        .collab-modal__close { position:absolute; top:10px; inset-inline-end:10px; background:#1e1e1e; border:0; color:#fff; width:36px; height:36px; border-radius:10px; cursor:pointer; z-index:2; }
        .collab-modal__header { display:flex; align-items:center; gap:16px; padding:28px 24px 0 24px; position: relative; z-index: 1; }
        .collab-modal__logo-wrap { 
            width:96px; 
            height:96px; 
            border-radius:16px; 
            background: linear-gradient(135deg, #2a2a2a 0%, #1a1a1a 50%, #0a0a0a 100%); 
            display:flex; 
            align-items:center; 
            justify-content:center; 
            overflow:hidden; 
            box-shadow: 
                0 8px 32px rgba(0,0,0,0.4),
                inset 0 2px 8px rgba(255,255,255,0.1),
                inset 0 -2px 8px rgba(0,0,0,0.3);
            position: relative;
        }
        .collab-modal__logo-wrap::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            border-radius: 16px;
            background: linear-gradient(135deg, rgba(255,255,255,0.1) 0%, transparent 50%, rgba(0,0,0,0.2) 100%);
            pointer-events: none;
        }
        .collab-modal__logo { 
            width:86px; 
            height:86px; 
            object-fit:contain; 
            border-radius:12px; 
            filter: drop-shadow(0 4px 12px rgba(0,0,0,0.3));
            transform: perspective(200px) rotateY(-5deg) rotateX(5deg);
            transition: transform 0.3s ease, filter 0.3s ease;
        }
        .collab-modal__logo:hover {
            transform: perspective(200px) rotateY(0deg) rotateX(0deg) scale(1.05);
            filter: drop-shadow(0 8px 24px rgba(0,0,0,0.4));
        }
        .collab-modal__company { font-size: 40px; font-weight:800; margin:0; }
        .collab-modal__role { margin-top:6px; opacity:.9; }
        .collab-modal__body { 
            display:grid; 
            grid-template-columns: 1.2fr .8fr; 
            gap:24px; 
            padding:24px 32px 32px 32px; 
            position: relative; 
            z-index: 1; 
        }
        
        /* Description Box - Glassmorphism Style */
        .collab-modal__desc { 
            line-height:1.7; 
            color:#f8f9fa; 
            font-size: 15px;
            background: rgba(255, 255, 255, 0.03);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 20px;
            padding: 20px;
            position: relative;
            overflow: hidden;
            box-shadow: 
                0 8px 32px rgba(0, 0, 0, 0.3),
                inset 0 1px 0 rgba(255, 255, 255, 0.1);
            height: 340px;
            overflow: hidden;
        }
        .collab-modal__desc::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: conic-gradient(
                from 0deg,
                transparent,
                rgba(79, 140, 255, 0.1),
                transparent,
                rgba(110, 231, 183, 0.1),
                transparent,
                rgba(255, 180, 58, 0.1),
                transparent
            );
            animation: rotate-gradient 8s linear infinite;
            opacity: 0.5;
        }
        @keyframes rotate-gradient {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        
        /* Facts Container - Modern Grid */
        .collab-modal__facts { 
            list-style:none; 
            margin:0; 
            padding:0; 
            display:grid; 
            gap:20px; 
        }
        
        /* Individual Fact Items - Neumorphism Style */
        .collab-modal__facts li { 
            display:flex; 
            align-items:center; 
            gap:14px; 
            background: linear-gradient(145deg, #1e1e1e, #2a2a2a);
            padding:16px 20px; 
            border-radius:18px; 
            border: none;
            box-shadow: 
                6px 6px 12px rgba(0, 0, 0, 0.4),
                -6px -6px 12px rgba(255, 255, 255, 0.05),
                inset 2px 2px 4px rgba(255, 255, 255, 0.05),
                inset -2px -2px 4px rgba(0, 0, 0, 0.3);
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
        }
        .collab-modal__facts li::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, 
                transparent 0%, 
                rgba(79, 140, 255, 0.1) 25%, 
                rgba(110, 231, 183, 0.1) 50%, 
                rgba(255, 180, 58, 0.1) 75%, 
                transparent 100%);
            transform: translateX(-100%);
            transition: transform 0.6s ease;
        }
        .collab-modal__facts li:hover::before {
            transform: translateX(100%);
        }
        .collab-modal__facts li:hover {
            transform: translateY(-4px) scale(1.02);
            box-shadow: 
                12px 12px 24px rgba(0, 0, 0, 0.5),
                -12px -12px 24px rgba(255, 255, 255, 0.08),
                inset 2px 2px 4px rgba(255, 255, 255, 0.08),
                inset -2px -2px 4px rgba(0, 0, 0, 0.2);
        }
        
        /* Icon Styling - Floating Orbs */
        .collab-modal__facts i { 
            color:#4f8cff; 
            font-size:16px; 
            background: radial-gradient(circle, rgba(79, 140, 255, 0.2) 0%, rgba(79, 140, 255, 0.1) 70%);
            padding: 10px;
            border-radius: 50%;
            min-width: 38px;
            height: 38px;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 
                0 3px 12px rgba(79, 140, 255, 0.3),
                inset 0 1px 3px rgba(255, 255, 255, 0.2);
            border: 1px solid rgba(79, 140, 255, 0.3);
            transition: all 0.3s ease;
        }
        .collab-modal__facts li:hover i {
            transform: scale(1.1) rotate(5deg);
            box-shadow: 
                0 8px 24px rgba(79, 140, 255, 0.4),
                inset 0 2px 4px rgba(255, 255, 255, 0.3);
        }
        
        /* Text Styling */
        .collab-modal__facts span {
            font-weight: 500;
            color: #f0f0f0;
            font-size: 15px;
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.3);
        }
        
        /* Actions Container - Modern Layout */
        .collab-modal__actions { 
            margin-top:24px; 
            display: flex;
            gap: 16px;
            flex-wrap: wrap;
        }
        html.skill-modal-open { overflow: hidden; }
        .collab-modal__dialog { max-height: 92vh; overflow: auto; }
        @media (max-width: 900px) {
            .collab-modal__company { font-size:32px; }
        }
        @media (max-width: 700px) {
            .collab-modal__dialog { width: 96vw; max-height: 92vh; border-radius: 16px; margin: 0; }
            .collab-modal__header { flex-direction: column; align-items: center; text-align: center; padding: 20px 16px 0 16px; }
            .collab-modal__logo-wrap { 
                width: 84px; 
                height: 84px; 
                border-radius: 14px;
                box-shadow: 
                    0 6px 24px rgba(0,0,0,0.4),
                    inset 0 2px 6px rgba(255,255,255,0.1),
                    inset 0 -2px 6px rgba(0,0,0,0.3);
            }
            .collab-modal__logo { 
                width: 76px; 
                height: 76px; 
                border-radius: 10px;
                transform: perspective(150px) rotateY(-3deg) rotateX(3deg);
            }
            .collab-modal__logo:hover {
                transform: perspective(150px) rotateY(0deg) rotateX(0deg) scale(1.03);
            }
            .collab-modal__body { 
                grid-template-columns: 1fr; 
                padding: 20px 16px 24px 16px; 
                gap: 20px;
            }
            .collab-modal__desc {
                padding: 20px;
                font-size: 15px;
            }
            .collab-modal__facts { 
                grid-template-columns: 1fr; 
                gap: 16px;
            }
            .collab-modal__facts li {
                padding: 16px 20px;
                gap: 14px;
            }
            .collab-modal__facts i {
                padding: 10px;
                min-width: 38px;
                height: 38px;
                font-size: 16px;
            }
            .collab-modal__actions { 
                margin-top: 20px; 
                flex-direction: column;
            }
            .collab-modal__actions .collaboration-cta { 
                width: 100%; 
                text-align: center; 
            }
            .collab-modal__company { font-size:28px; }
        }
        @media (max-width: 400px) {
            .collab-modal__company { font-size:24px; }
            .collab-modal__close { width: 40px; height: 40px; }
        }
        </style>
        <script>
        (function(){
            var root = document.getElementById('skill-collab-<?php echo esc_js($widget_id); ?>');
            if(!root) return;
            
            // Initialize modals with display: flex
            root.querySelectorAll('.collab-modal').forEach(function(modal){
                modal.style.display = 'flex';
            });
            
            root.addEventListener('click', function(e){
                var item = e.target.closest('.js-open-collab-modal');
                if(!item || !root.contains(item)) return;
                var id = item.getAttribute('data-modal-id');
                if(!id) return;
                var modal = document.getElementById(id);
                if(!modal) return;
                modal.classList.add('is-open');
                document.documentElement.classList.add('skill-modal-open');
            });
            root.querySelectorAll('.collab-modal').forEach(function(modal){
                modal.addEventListener('click', function(ev){
                    if(ev.target.closest('[data-close]')){ 
                        modal.classList.remove('is-open'); 
                        document.documentElement.classList.remove('skill-modal-open'); 
                    }
                });
            });
            document.addEventListener('keydown', function(ev){
                if(ev.key === 'Escape'){
                    root.querySelectorAll('.collab-modal.is-open').forEach(function(m){ 
                        m.classList.remove('is-open'); 
                    });
                    document.documentElement.classList.remove('skill-modal-open');
                }
            });
        })();
        </script>
        <?php
    }
}